/*
 * Decompiled with CFR 0.152.
 */
package overflowdb.storage;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import org.h2.mvstore.FileStore;
import org.h2.mvstore.MVMap;
import org.h2.mvstore.MVStore;
import overflowdb.storage.BackwardsCompatibilityError;
import overflowdb.util.StringInterner;

public class OdbStorage
implements AutoCloseable {
    public static final int STORAGE_FORMAT_VERSION = 2;
    public static final String METADATA_KEY_STORAGE_FORMAT_VERSION = "STORAGE_FORMAT_VERSION";
    public static final String METADATA_KEY_STRING_TO_INT_MAX_ID = "STRING_TO_INT_MAX_ID";
    public static final String METADATA_KEY_LIBRARY_VERSIONS_MAX_ID = "LIBRARY_VERSIONS_MAX_ID";
    public static final String METADATA_PREFIX_LIBRARY_VERSIONS = "LIBRARY_VERSIONS_ENTRY_";
    private static final String INDEX_PREFIX = "index_";
    private final File mvstoreFile;
    private final StringInterner stringInterner;
    private FileStore mvstoreFileStore;
    protected MVStore mvstore;
    private MVMap<Long, byte[]> nodesMVMap;
    private MVMap<String, String> metadataMVMap;
    private MVMap<String, Integer> stringToIntMappings;
    private boolean closed;
    private final AtomicInteger stringToIntMappingsMaxId = new AtomicInteger(0);
    private ArrayList<String> stringToIntReverseMappings;
    private int libraryVersionsIdCurrentRun;

    public static OdbStorage createWithTempFile(StringInterner stringInterner) {
        return new OdbStorage(Optional.empty(), stringInterner);
    }

    public static OdbStorage createWithSpecificLocation(File mvstoreFile, StringInterner stringInterner) {
        return new OdbStorage(Optional.ofNullable(mvstoreFile), stringInterner);
    }

    private OdbStorage(Optional<File> mvstoreFileMaybe, StringInterner stringInterner) {
        this.stringInterner = stringInterner;
        if (mvstoreFileMaybe.isPresent()) {
            this.mvstoreFile = mvstoreFileMaybe.get();
            if (this.mvstoreFile.exists() && this.mvstoreFile.length() > 0L) {
                this.verifyStorageVersion();
                this.initializeStringToIntMaxId();
            }
        } else {
            try {
                this.mvstoreFile = File.createTempFile("mvstore", ".bin");
                if (!System.getProperty("os.name").toLowerCase().contains("win")) {
                    this.mvstoreFileStore = new FileStore();
                    boolean readOnly = false;
                    char[] encryptionKey = null;
                    this.mvstoreFileStore.open(this.mvstoreFile.getAbsolutePath(), readOnly, encryptionKey);
                    this.mvstoreFile.delete();
                } else {
                    this.mvstoreFile.deleteOnExit();
                }
            }
            catch (IOException e) {
                throw new RuntimeException("cannot create tmp file for mvstore", e);
            }
        }
    }

    private void initializeStringToIntMaxId() {
        MVMap<String, String> metadata = this.getMetaDataMVMap();
        if (metadata.containsKey((Object)METADATA_KEY_STRING_TO_INT_MAX_ID)) {
            int maxIndexFromStorage = Integer.parseInt((String)metadata.get((Object)METADATA_KEY_STRING_TO_INT_MAX_ID));
            this.stringToIntMappingsMaxId.set(maxIndexFromStorage);
        }
    }

    private void verifyStorageVersion() {
        this.ensureMVStoreAvailable();
        MVMap<String, String> metaData = this.getMetaDataMVMap();
        if (!metaData.containsKey((Object)METADATA_KEY_STORAGE_FORMAT_VERSION)) {
            throw new BackwardsCompatibilityError("storage metadata does not contain version number, this must be an old format.");
        }
        String storageFormatVersionString = (String)metaData.get((Object)METADATA_KEY_STORAGE_FORMAT_VERSION);
        int storageFormatVersion = Integer.parseInt(storageFormatVersionString);
        if (storageFormatVersion != 2) {
            throw new BackwardsCompatibilityError(String.format("attempting to open storage with different version: %s; this version of overflowdb requires the version to be exactly %s", storageFormatVersion, 2));
        }
    }

    public void persist(long id, byte[] node) {
        if (!this.closed) {
            this.getNodesMVMap().put((Object)id, (Object)node);
        }
    }

    public void flush() {
        if (this.mvstore != null) {
            this.getMetaDataMVMap().put((Object)METADATA_KEY_STORAGE_FORMAT_VERSION, (Object)String.format("%s", 2));
            this.getMetaDataMVMap().put((Object)METADATA_KEY_STRING_TO_INT_MAX_ID, (Object)String.format("%s", this.stringToIntMappingsMaxId.get()));
            this.mvstore.commit();
        }
    }

    @Override
    public void close() {
        this.closed = true;
        this.flush();
        if (this.mvstore != null) {
            this.mvstore.close();
        }
    }

    public File getStorageFile() {
        return this.mvstoreFile;
    }

    public void removeNode(Long id) {
        this.getNodesMVMap().remove((Object)id);
    }

    public Set<Map.Entry<Long, byte[]>> allNodes() {
        return this.getNodesMVMap().entrySet();
    }

    public synchronized MVMap<Long, byte[]> getNodesMVMap() {
        this.ensureMVStoreAvailable();
        if (this.nodesMVMap == null) {
            this.nodesMVMap = this.mvstore.openMap("nodes");
        }
        return this.nodesMVMap;
    }

    public synchronized MVMap<String, String> getMetaDataMVMap() {
        this.ensureMVStoreAvailable();
        if (this.metadataMVMap == null) {
            this.metadataMVMap = this.mvstore.openMap("metadata");
        }
        return this.metadataMVMap;
    }

    public synchronized MVMap<String, Integer> getStringToIntMappings() {
        this.ensureMVStoreAvailable();
        if (this.stringToIntMappings == null) {
            this.stringToIntMappings = this.mvstore.openMap("stringToIntMappings");
        }
        if (this.stringToIntReverseMappings == null) {
            int mappingsCount = this.stringToIntMappings.size();
            this.stringToIntReverseMappings = new ArrayList(mappingsCount);
            this.stringToIntMappings.forEach((string, id) -> {
                this.ensureCapacity(this.stringToIntReverseMappings, id + 1);
                this.stringToIntReverseMappings.set((int)id, (String)string);
            });
        }
        return this.stringToIntMappings;
    }

    public int lookupOrCreateStringToIntMapping(String s) {
        MVMap<String, Integer> mappings = this.getStringToIntMappings();
        if (mappings.containsKey((Object)s)) {
            return (Integer)mappings.get((Object)s);
        }
        return this.createStringToIntMapping(s);
    }

    private int createStringToIntMapping(String s) {
        int index = this.stringToIntMappingsMaxId.incrementAndGet();
        this.getStringToIntMappings().put((Object)s, (Object)index);
        this.ensureCapacity(this.stringToIntReverseMappings, index + 1);
        this.stringToIntReverseMappings.set(index, s);
        return index;
    }

    private void ensureCapacity(ArrayList array, int requiredMinSize) {
        while (array.size() < requiredMinSize) {
            array.add(null);
        }
    }

    public String reverseLookupStringToIntMapping(int stringId) {
        this.getStringToIntMappings();
        String string = this.stringToIntReverseMappings.get(stringId);
        return this.stringInterner.intern(string);
    }

    private void ensureMVStoreAvailable() {
        if (this.mvstore == null) {
            this.mvstore = this.initializeMVStore();
            this.persistOdbLibraryVersion();
            this.libraryVersionsIdCurrentRun = this.initializeLibraryVersionsIdCurrentRun();
        }
    }

    private MVStore initializeMVStore() {
        MVStore.Builder builder = new MVStore.Builder().autoCommitBufferSize(8192).compress().autoCommitDisabled();
        if (this.mvstoreFileStore != null) {
            builder.fileStore(this.mvstoreFileStore);
        } else {
            builder.fileName(this.mvstoreFile.getAbsolutePath());
        }
        return builder.open();
    }

    private int initializeLibraryVersionsIdCurrentRun() {
        MVMap<String, String> metaData = this.getMetaDataMVMap();
        int res = metaData.containsKey((Object)METADATA_KEY_LIBRARY_VERSIONS_MAX_ID) ? Integer.parseInt((String)metaData.get((Object)METADATA_KEY_LIBRARY_VERSIONS_MAX_ID)) + 1 : 0;
        metaData.put((Object)METADATA_KEY_LIBRARY_VERSIONS_MAX_ID, (Object)("" + res));
        return res;
    }

    private Map<String, String> getIndexNameMap(MVStore store) {
        return store.getMapNames().stream().filter(s -> s.startsWith(INDEX_PREFIX)).collect(Collectors.toConcurrentMap(this::removeIndexPrefix, s -> s));
    }

    public Set<String> getIndexNames() {
        return this.getIndexNameMap(this.mvstore).keySet();
    }

    private String removeIndexPrefix(String s) {
        assert (s.startsWith(INDEX_PREFIX));
        return s.substring(INDEX_PREFIX.length());
    }

    public MVMap<Object, long[]> openIndex(String indexName) {
        String mapName = this.getIndexMapName(indexName);
        return this.mvstore.openMap(mapName);
    }

    private String getIndexMapName(String indexName) {
        return INDEX_PREFIX + indexName;
    }

    public void clearIndices() {
        this.getIndexNames().forEach(this::clearIndex);
    }

    public void clearIndex(String indexName) {
        this.openIndex(indexName).clear();
    }

    public byte[] getSerializedNode(long nodeId) {
        return (byte[])this.getNodesMVMap().get((Object)nodeId);
    }

    private void persistOdbLibraryVersion() {
        Class<?> clazz = this.getClass();
        String version = clazz.getPackage().getImplementationVersion();
        if (version != null) {
            this.persistLibraryVersion(clazz.getCanonicalName(), version);
        }
    }

    public void persistLibraryVersion(String name, String version) {
        String key = String.format("%s%d_%s", METADATA_PREFIX_LIBRARY_VERSIONS, this.libraryVersionsIdCurrentRun, name);
        this.getMetaDataMVMap().put((Object)key, (Object)version);
    }

    public ArrayList<Map<String, String>> getAllLibraryVersions() {
        HashMap libraryVersionsByRunId = new HashMap();
        this.getMetaDataMVMap().forEach((key, version) -> {
            if (key.startsWith(METADATA_PREFIX_LIBRARY_VERSIONS)) {
                String withoutPrefix = key.substring(METADATA_PREFIX_LIBRARY_VERSIONS.length());
                int firstDividerIndex = withoutPrefix.indexOf(95);
                int runId = Integer.parseInt(withoutPrefix.substring(0, firstDividerIndex));
                String library = withoutPrefix.substring(firstDividerIndex + 1);
                Map versionInfos = libraryVersionsByRunId.computeIfAbsent(runId, i -> new HashMap());
                versionInfos.put(library, version);
            }
        });
        return new ArrayList<Map<String, String>>(libraryVersionsByRunId.values());
    }
}

