/*
 * Decompiled with CFR 0.152.
 */
package org.opendataloader.pdf.html;

import java.awt.image.BufferedImage;
import java.awt.image.RenderedImage;
import java.io.Closeable;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.imageio.ImageIO;
import org.opendataloader.pdf.api.Config;
import org.opendataloader.pdf.containers.StaticLayoutContainers;
import org.verapdf.wcag.algorithms.entities.IObject;
import org.verapdf.wcag.algorithms.entities.SemanticHeading;
import org.verapdf.wcag.algorithms.entities.SemanticParagraph;
import org.verapdf.wcag.algorithms.entities.SemanticTextNode;
import org.verapdf.wcag.algorithms.entities.content.ImageChunk;
import org.verapdf.wcag.algorithms.entities.geometry.BoundingBox;
import org.verapdf.wcag.algorithms.entities.lists.ListItem;
import org.verapdf.wcag.algorithms.entities.lists.PDFList;
import org.verapdf.wcag.algorithms.entities.tables.tableBorders.TableBorder;
import org.verapdf.wcag.algorithms.entities.tables.tableBorders.TableBorderCell;
import org.verapdf.wcag.algorithms.entities.tables.tableBorders.TableBorderRow;
import org.verapdf.wcag.algorithms.semanticalgorithms.consumers.ContrastRatioConsumer;
import org.verapdf.wcag.algorithms.semanticalgorithms.containers.StaticContainers;

public class HtmlGenerator
implements Closeable {
    protected static final Logger LOGGER = Logger.getLogger(HtmlGenerator.class.getCanonicalName());
    protected final FileWriter htmlWriter;
    protected final String pdfFileName;
    protected final Path pdfFilePath;
    protected final String figureDirName;
    protected final Path figureDirPath;
    protected ContrastRatioConsumer contrastRatioConsumer;
    protected final String htmlFileName;
    protected final Path htmlFilePath;
    protected int tableNesting = 0;
    private final String password;

    public HtmlGenerator(File inputPdf, String outputFolder, Config config) throws IOException {
        this.pdfFileName = inputPdf.getName();
        this.pdfFilePath = inputPdf.toPath().toAbsolutePath();
        this.password = config.getPassword();
        this.htmlFileName = this.pdfFileName.substring(0, this.pdfFileName.length() - 3) + "html";
        this.htmlFilePath = Path.of(outputFolder, this.htmlFileName);
        this.figureDirName = this.pdfFileName.substring(0, this.pdfFileName.length() - 4) + "_figures";
        this.figureDirPath = Path.of(outputFolder, this.figureDirName);
        this.htmlWriter = new FileWriter(this.htmlFilePath.toFile(), StandardCharsets.UTF_8);
    }

    public void writeToHtml(List<List<IObject>> contents) {
        try {
            this.htmlWriter.write("<!DOCTYPE html>\n");
            this.htmlWriter.write("<html lang=\"und\">\n<head>\n<meta charset=\"utf-8\">\n");
            this.htmlWriter.write("<title>" + this.pdfFileName + "</title>\n");
            this.htmlWriter.write("</head>\n<body>\n");
            for (int pageNumber = 0; pageNumber < StaticContainers.getDocument().getNumberOfPages(); ++pageNumber) {
                for (IObject content : contents.get(pageNumber)) {
                    this.write(content);
                }
            }
            this.htmlWriter.write("\n</body>\n</html>");
            LOGGER.log(Level.INFO, "Created {0}", this.htmlFilePath);
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Unable to create html output: " + e.getMessage());
        }
    }

    protected void write(IObject object) throws IOException {
        if (object instanceof ImageChunk) {
            this.writeImage((ImageChunk)object);
        } else if (object instanceof SemanticHeading) {
            this.writeHeading((SemanticHeading)object);
        } else if (object instanceof SemanticParagraph) {
            this.writeParagraph((SemanticParagraph)object);
        } else if (object instanceof SemanticTextNode) {
            this.writeSemanticTextNode((SemanticTextNode)object);
        } else if (object instanceof TableBorder) {
            this.writeTable((TableBorder)object);
        } else if (object instanceof PDFList) {
            this.writeList((PDFList)object);
        } else {
            return;
        }
        if (!this.isInsideTable()) {
            this.htmlWriter.write("\n");
        }
    }

    protected void writeImage(ImageChunk image) throws IOException {
        String figureFileName;
        Path figureFilePath;
        boolean isFileCreated;
        int currentImageIndex = StaticLayoutContainers.incrementImageIndex();
        if (currentImageIndex == 1) {
            this.figureDirPath.toFile().mkdirs();
            this.contrastRatioConsumer = new ContrastRatioConsumer(this.pdfFilePath.toString(), this.password, false, null);
        }
        if (isFileCreated = this.createImageFile(image, (figureFilePath = this.figureDirPath.resolve(figureFileName = String.format("figure%d.png", currentImageIndex))).toString())) {
            String relativePathName = figureFilePath.subpath(this.figureDirPath.getNameCount() - 1, this.figureDirPath.getNameCount() + 1).toString().replace("\\", "/");
            String imageString = String.format("<img src=\"%s\" alt=\"figure%d\">", relativePathName, currentImageIndex);
            this.htmlWriter.write(imageString);
            this.htmlWriter.write("\n");
        }
    }

    protected boolean createImageFile(ImageChunk image, String fileName) {
        try {
            BoundingBox imageBox = image.getBoundingBox();
            BufferedImage targetImage = this.contrastRatioConsumer.getPageSubImage(imageBox);
            if (targetImage == null) {
                return false;
            }
            File outputFile = new File(fileName);
            ImageIO.write((RenderedImage)targetImage, "png", outputFile);
            return true;
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Unable to create image files: " + e.getMessage());
            return false;
        }
    }

    protected void writeList(PDFList list) throws IOException {
        this.htmlWriter.write("<ul>");
        this.htmlWriter.write("\n");
        for (ListItem item : list.getListItems()) {
            this.htmlWriter.write("<li>");
            this.htmlWriter.write("<p>");
            this.htmlWriter.write(this.getCorrectString(item.toString()));
            this.htmlWriter.write("</p>");
            for (IObject object : item.getContents()) {
                this.write(object);
            }
            this.htmlWriter.write("</li>");
            this.htmlWriter.write("\n");
        }
        this.htmlWriter.write("</ul>");
        this.htmlWriter.write("\n");
    }

    protected void writeSemanticTextNode(SemanticTextNode textNode) throws IOException {
        this.htmlWriter.write("<figcaption>");
        this.htmlWriter.write(this.getCorrectString(textNode.getValue()));
        this.htmlWriter.write("</figcaption>");
        this.htmlWriter.write("\n");
    }

    protected void writeTable(TableBorder table) throws IOException {
        this.enterTable();
        this.htmlWriter.write("<table border=\"1\">");
        this.htmlWriter.write("\n");
        for (int rowNumber = 0; rowNumber < table.getNumberOfRows(); ++rowNumber) {
            TableBorderRow row = table.getRow(rowNumber);
            this.htmlWriter.write("<tr>");
            this.htmlWriter.write("\n");
            for (int colNumber = 0; colNumber < table.getNumberOfColumns(); ++colNumber) {
                TableBorderCell cell = row.getCell(colNumber);
                if (cell.getRowNumber() != rowNumber || cell.getColNumber() != colNumber) continue;
                boolean isHeader = rowNumber == 0;
                this.writeCellTag(cell, isHeader);
                List<IObject> contents = cell.getContents();
                if (!contents.isEmpty()) {
                    for (IObject contentItem : contents) {
                        this.write(contentItem);
                    }
                }
                if (isHeader) {
                    this.htmlWriter.write("</th>");
                } else {
                    this.htmlWriter.write("</td>");
                }
                this.htmlWriter.write("\n");
            }
            this.htmlWriter.write("</tr>");
            this.htmlWriter.write("\n");
        }
        this.htmlWriter.write("</table>");
        this.htmlWriter.write("\n");
        this.leaveTable();
    }

    protected void writeParagraph(SemanticParagraph paragraph) throws IOException {
        String paragraphValue = paragraph.getValue();
        double paragraphIndent = paragraph.getColumns().get(0).getBlocks().get(0).getFirstLineIndent();
        this.htmlWriter.write("<p>");
        if (paragraphIndent > 0.0) {
            this.htmlWriter.write("");
        }
        if (this.isInsideTable() && StaticContainers.isKeepLineBreaks()) {
            paragraphValue = paragraphValue.replace("\n", "<br>");
        }
        this.htmlWriter.write(this.getCorrectString(paragraphValue));
        this.htmlWriter.write("</p>");
        this.htmlWriter.write("\n");
    }

    protected void writeHeading(SemanticHeading heading) throws IOException {
        int headingLevel = Math.min(6, Math.max(1, heading.getHeadingLevel()));
        this.htmlWriter.write("<h" + headingLevel + ">");
        this.htmlWriter.write(this.getCorrectString(heading.getValue()));
        this.htmlWriter.write("</h" + headingLevel + ">");
        this.htmlWriter.write("\n");
    }

    private void writeCellTag(TableBorderCell cell, boolean isHeader) throws IOException {
        int rowSpan;
        String tag = isHeader ? "<th" : "<td";
        StringBuilder cellTag = new StringBuilder(tag);
        int colSpan = cell.getColSpan();
        if (colSpan != 1) {
            cellTag.append(" colspan=\"").append(colSpan).append("\"");
        }
        if ((rowSpan = cell.getRowSpan()) != 1) {
            cellTag.append(" rowspan=\"").append(rowSpan).append("\"");
        }
        cellTag.append(">");
        this.htmlWriter.write(this.getCorrectString(cellTag.toString()));
    }

    protected void enterTable() {
        ++this.tableNesting;
    }

    protected void leaveTable() {
        if (this.tableNesting > 0) {
            --this.tableNesting;
        }
    }

    protected boolean isInsideTable() {
        return this.tableNesting > 0;
    }

    protected String getCorrectString(String value) {
        if (value != null) {
            return value.replace("\u0000", "");
        }
        return null;
    }

    @Override
    public void close() throws IOException {
        if (this.htmlWriter != null) {
            this.htmlWriter.close();
        }
        if (this.contrastRatioConsumer != null) {
            this.contrastRatioConsumer.close();
        }
    }
}

