/*
 * Decompiled with CFR 0.152.
 */
package org.opendataloader.pdf.text;

import java.io.Closeable;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import org.verapdf.wcag.algorithms.entities.IObject;
import org.verapdf.wcag.algorithms.entities.SemanticHeading;
import org.verapdf.wcag.algorithms.entities.SemanticParagraph;
import org.verapdf.wcag.algorithms.entities.SemanticTextNode;
import org.verapdf.wcag.algorithms.entities.lists.ListItem;
import org.verapdf.wcag.algorithms.entities.lists.PDFList;
import org.verapdf.wcag.algorithms.entities.tables.tableBorders.TableBorder;
import org.verapdf.wcag.algorithms.entities.tables.tableBorders.TableBorderRow;

public class TextGenerator
implements Closeable {
    private static final Logger LOGGER = Logger.getLogger(TextGenerator.class.getCanonicalName());
    private static final String INDENT = "  ";
    private final FileWriter textWriter;
    private final String textFileName;
    private final String lineSeparator = System.lineSeparator();

    public TextGenerator(File inputPdf, String outputFolder) throws IOException {
        String cutPdfFileName = inputPdf.getName();
        this.textFileName = outputFolder + File.separator + cutPdfFileName.substring(0, cutPdfFileName.length() - 3) + "txt";
        this.textWriter = new FileWriter(this.textFileName, StandardCharsets.UTF_8);
    }

    public void writeToText(List<List<IObject>> contents) {
        try {
            for (int pageIndex = 0; pageIndex < contents.size(); ++pageIndex) {
                List<IObject> pageContents = contents.get(pageIndex);
                this.writeContents(pageContents, 0);
                if (pageIndex >= contents.size() - 1) continue;
                this.textWriter.write(this.lineSeparator);
            }
            LOGGER.log(Level.INFO, "Created {0}", this.textFileName);
        }
        catch (IOException e) {
            LOGGER.log(Level.WARNING, "Unable to create text output: " + e.getMessage());
        }
    }

    private void writeContents(List<IObject> contents, int indentLevel) throws IOException {
        for (int index = 0; index < contents.size(); ++index) {
            this.write(contents.get(index), indentLevel);
            if (index >= contents.size() - 1) continue;
            this.textWriter.write(this.lineSeparator);
        }
    }

    private void write(IObject object, int indentLevel) throws IOException {
        if (object instanceof SemanticHeading) {
            this.writeMultiline(((SemanticHeading)object).getValue(), indentLevel);
        } else if (object instanceof SemanticParagraph) {
            this.writeMultiline(((SemanticParagraph)object).getValue(), indentLevel);
        } else if (object instanceof SemanticTextNode) {
            this.writeMultiline(((SemanticTextNode)object).getValue(), indentLevel);
        } else if (object instanceof PDFList) {
            this.writeList((PDFList)object, indentLevel);
        } else if (object instanceof TableBorder) {
            this.writeTable((TableBorder)object, indentLevel);
        }
    }

    private void writeList(PDFList list, int indentLevel) throws IOException {
        for (ListItem item : list.getListItems()) {
            String indent = this.indent(indentLevel);
            String itemText = this.compactWhitespace(this.collectPlainText(item.getContents()));
            if (!itemText.isEmpty()) {
                this.textWriter.write(indent);
                this.textWriter.write(itemText);
                this.textWriter.write(this.lineSeparator);
            }
            if (item.getContents().isEmpty()) continue;
            this.writeContents(item.getContents(), indentLevel + 1);
        }
    }

    private void writeTable(TableBorder table, int indentLevel) throws IOException {
        String indent = this.indent(indentLevel);
        for (TableBorderRow row : table.getRows()) {
            String rowText = Arrays.stream(row.getCells()).map(cell -> this.compactWhitespace(this.collectPlainText(cell.getContents()))).filter(text -> !text.isEmpty()).collect(Collectors.joining("\t"));
            if (rowText.isEmpty()) continue;
            this.textWriter.write(indent);
            this.textWriter.write(rowText);
            this.textWriter.write(this.lineSeparator);
        }
    }

    private String collectPlainText(List<IObject> contents) {
        StringBuilder builder = new StringBuilder();
        for (IObject content : contents) {
            String piece = this.extractPlainText(content);
            if (piece.isEmpty()) continue;
            if (builder.length() > 0) {
                builder.append(' ');
            }
            builder.append(piece);
        }
        return builder.toString();
    }

    private String extractPlainText(IObject content) {
        if (content instanceof SemanticHeading) {
            return this.sanitize(((SemanticHeading)content).getValue());
        }
        if (content instanceof SemanticParagraph) {
            return this.sanitize(((SemanticParagraph)content).getValue());
        }
        if (content instanceof SemanticTextNode) {
            return this.sanitize(((SemanticTextNode)content).getValue());
        }
        if (content instanceof PDFList) {
            PDFList list = (PDFList)content;
            return list.getListItems().stream().map(item -> this.compactWhitespace(this.collectPlainText(item.getContents()))).filter(text -> !text.isEmpty()).collect(Collectors.joining(" "));
        }
        if (content instanceof TableBorder) {
            TableBorder table = (TableBorder)content;
            return Arrays.stream(table.getRows()).map(row -> Arrays.stream(row.getCells()).map(cell -> this.compactWhitespace(this.collectPlainText(cell.getContents()))).filter(text -> !text.isEmpty()).collect(Collectors.joining(" "))).filter(text -> !text.isEmpty()).collect(Collectors.joining(" "));
        }
        return "";
    }

    private void writeMultiline(String value, int indentLevel) throws IOException {
        String[] lines;
        if (value == null) {
            return;
        }
        String sanitized = this.sanitize(value);
        String indent = this.indent(indentLevel);
        for (String line : lines = sanitized.split("\r?\n", -1)) {
            if (line.isBlank()) continue;
            this.textWriter.write(indent);
            this.textWriter.write(line);
            this.textWriter.write(this.lineSeparator);
        }
    }

    private String indent(int level) {
        if (level <= 0) {
            return "";
        }
        return INDENT.repeat(level);
    }

    private String sanitize(String value) {
        return value == null ? "" : value.replace("\u0000", " ");
    }

    private String compactWhitespace(String value) {
        if (value == null) {
            return "";
        }
        String sanitized = this.sanitize(value);
        return sanitized.replaceAll("\\s+", " ").trim();
    }

    @Override
    public void close() throws IOException {
        if (this.textWriter != null) {
            this.textWriter.close();
        }
    }
}

