/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */


jstestdriver.convertToJson = function(delegate) {
  var serialize = jstestdriver.parameterSerialize
  return function(url, data, callback, type) {
    delegate(url, serialize(data), callback, type);
  };
};


jstestdriver.parameterSerialize = function(data) {
  var modifiedData = {};
  for (var key in data) {
    modifiedData[key] = JSON.stringify(data[key]);
  }
  return modifiedData;
};


jstestdriver.bind = function(context, func) {
  function bound() {
    return func.apply(context, arguments);
  };
  bound.toString = function() {
    return "bound: " + context + " to: " + func;
  }
  return bound;
};


jstestdriver.extractId = function(url) {
  return url.match(/\/id\/(\d+)\//)[1];
};


jstestdriver.createPath = function(basePath, path) {
  var prefix = basePath.match(/^(.*)\/(slave|runner|bcr)\//)[1];
  return prefix + path;
};


jstestdriver.getBrowserFriendlyName = function() {
  if (jstestdriver.jQuery.browser.safari) {
    if (navigator.userAgent.indexOf('Chrome') != -1) {
      return 'Chrome';
    }
    return 'Safari';
  } else if (jstestdriver.jQuery.browser.opera) {
    return 'Opera';
  } else if (jstestdriver.jQuery.browser.msie) {
    return 'Internet Explorer';
  } else if (jstestdriver.jQuery.browser.mozilla) {
    if (navigator.userAgent.indexOf('Firefox') != -1) {
      return 'Firefox';
    }
    return 'Mozilla';
  }
};


jstestdriver.getBrowserFriendlyVersion = function() {
  if (jstestdriver.jQuery.browser.msie) {
    if (typeof XDomainRequest != 'undefined') {
      return '8.0';
    } 
  } else if (jstestdriver.jQuery.browser.safari) {
    if (navigator.appVersion.indexOf('Chrome/') != -1) {
      return navigator.appVersion.match(/Chrome\/(.*)\s/)[1];
    }
  }
  return jstestdriver.jQuery.browser.version;
};

jstestdriver.trim = function(str) {
  return str.replace(/(^\s*)|(\s*$)/g,'');
};


/**
 * Renders an html string as a dom nodes.
 * @param {string} htmlString The string to be rendered as html.
 * @param {Document} owningDocument The window that should own the html.
 */
jstestdriver.toHtml = function(htmlString, owningDocument) {
  var fragment = owningDocument.createDocumentFragment();
  var wrapper = owningDocument.createElement('div');
  wrapper.innerHTML = jstestdriver.trim(jstestdriver.stripHtmlComments(htmlString));
  while(wrapper.firstChild) {
    fragment.appendChild(wrapper.firstChild);
  }
  var ret =  fragment.childNodes.length > 1 ? fragment : fragment.firstChild;
  return ret;
};


jstestdriver.stripHtmlComments = function(htmlString) {
  var stripped = [];
  function getCommentIndices(offset) {
    var start = htmlString.indexOf('<!--', offset);
    var stop = htmlString.indexOf('-->', offset) + '-->'.length;
    if (start == -1) {
      return null;
    }
    return {
      'start' : start,
      'stop' : stop
    };
  }
  var offset = 0;
  while(true) {
    var comment = getCommentIndices(offset);
    if (!comment) {
      stripped.push(htmlString.slice(offset));
      break;
    }
    var frag = htmlString.slice(offset, comment.start);
    stripped.push(frag);
    offset = comment.stop;
  }
  return stripped.join('');
}


/**
 * Appends html string to the body.
 * @param {string} htmlString The string to be rendered as html.
 * @param {Document} owningDocument The window that should own the html.
 */
jstestdriver.appendHtml = function(htmlString, owningDocument) {
  var node = jstestdriver.toHtml(htmlString, owningDocument);
  jstestdriver.jQuery(owningDocument.body).append(node);
};


/**
 * @return {Number} The ms since the epoch.
 */
jstestdriver.now = function() { return new Date().getTime();}


/**
 * Creates a wrapper for jQuery.ajax that make a synchronous post
 * @param {jQuery} jQuery
 * @return {function(url, data):null}
 */
jstestdriver.createSynchPost = function(jQuery) {
  return jstestdriver.convertToJson(function(url, data) {
    return jQuery.ajax({
      'async' : false,
      'data' : data,
      'type' : 'POST',
      'url' : url
    });
  });
};

jstestdriver.utils = {};

/**
 * Checks to see if an object is a a certain native type.
 * @param instance An instance to check.
 * @param nativeType A string of the type expected.
 * @returns True if of that type.
 */
jstestdriver.utils.isNative = function(instance, nativeType) {
  try {
    var typeString = String(Object.prototype.toString.apply(instance));
    return typeString.toLowerCase().indexOf(nativeType.toLowerCase()) != -1;
  } catch (e) {
    return false;
  }
};

jstestdriver.utils.serializeErrors = function(errors) {
  var out = [];
  out.push('[');
  for (var i = 0; i < errors.length; ++i) {
    jstestdriver.utils.serializeErrorToArray(errors[i], out);
    if (i < errors.length - 1) {
      out.push(',');
    }
  }
  out.push(']');
  return out.join('');
};

jstestdriver.utils.serializeErrorToArray = function(error, out) {
  if (jstestdriver.utils.isNative(error, 'Error')) {
    out.push('{');
    out.push('"message":');
    this.serializeObjectToArray(error.message, out);
    this.serializePropertyOnObject('name', error, out);
    this.serializePropertyOnObject('description', error, out);
    this.serializePropertyOnObject('fileName', error, out);
    this.serializePropertyOnObject('lineNumber', error, out);
    this.serializePropertyOnObject('number', error, out);
    this.serializePropertyOnObject('stack', error, out);
    out.push('}');
  } else {
    out.push(jstestdriver.utils.serializeObject(error));
  }
};

jstestdriver.utils.serializeObject = function(obj) {
  var out = [];
  jstestdriver.utils.serializeObjectToArray(obj, out);
  return out.join('');
};


jstestdriver.utils.serializeObjectToArray =
   function(obj, opt_out){
  var out = opt_out || out;
  if (jstestdriver.utils.isNative(obj, 'Array')) {
    out.push('[');
    var arr = /** @type {Array.<Object>} */ obj;
    for ( var i = 0; i < arr.length; i++) {
      this.serializeObjectToArray(arr[i], out);
      if (i < arr.length - 1) {
        out.push(',');
      }
    }
    out.push(']');
  } else {
    var serial = jstestdriver.angular.toJson(obj);
    if (!serial.length) {
      serial = '["Bad serialization of ' + String(obj) + ':' +
          Object.prototype.toString.call(obj) + '"]';
    }
    out.push(serial);
  }
  return out;
};


jstestdriver.utils.serializePropertyOnObject = function(name, obj, out) {
  if (name in obj) {
    out.push(',');
    out.push('"' + name + '":');
    this.serializeObjectToArray(obj[name], out);
  }
};
/*
 * Copyright 2009 Google Inc.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 * 
 * http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
jstestdriver.Heartbeat = function(id,
                                  url,
                                  capturePath,
                                  sendRequest,
                                  interval,
                                  setTimeout,
                                  getTime,
                                  view,
                                  navigateToPath) {
  this.id_ = id;
  this.url_ = url;
  this.capturePath_ = capturePath;
  this.retries_ = 0;
  this.sendRequest_ = sendRequest;
  this.interval_ = interval;
  this.boundHeartbeatCallback_ = jstestdriver.bind(this, this.heartbeatCallback);
  this.boundSendHeartBeat_ = jstestdriver.bind(this, this.sendHeartbeat);
  this.boundErrorCallback_ = jstestdriver.bind(this, this.errorCallback);
  this.sent_ = 0;
  this.timeoutId_ = -1;
  this.setTimeout_ = setTimeout;
  this.getTime_ = getTime;
  this.view_ = view;
  this.navigateToPath_ = navigateToPath;
};


jstestdriver.Heartbeat.RETRY_LIMIT = 50;


jstestdriver.Heartbeat.prototype.start = function() {
  this.view_.updateConnected(true);
  this.sendHeartbeat();
};


jstestdriver.Heartbeat.prototype.stop = function() {
  jstestdriver.clearTimeout(this.timeoutId_);
  this.view_.updateStatus('Dead.');
};


jstestdriver.Heartbeat.prototype.sendHeartbeat = function() {
  this.sent_ = this.getTime_();
  this.view_.updateLastBeat(this.sent_);
  this.sendRequest_(this.url_, { id: this.id_ },
                    this.boundHeartbeatCallback_,
                    this.boundErrorCallback_);
};


jstestdriver.Heartbeat.prototype.errorCallback = function() {
  this.view_.updateConnected(false);
  if (this.retries_ > jstestdriver.Heartbeat.RETRY_LIMIT) {
    this.stop();
    return;
  }
  this.retries_++;
  this.view_.updateStatus('Retrying ' +
      this.retries_ + ' out of ' + jstestdriver.Heartbeat.RETRY_LIMIT);
  this.timeoutId_ = this.setTimeout_(this.boundSendHeartBeat_, this.interval_);
}

// TODO(corysmith): convert response to json.
jstestdriver.Heartbeat.prototype.heartbeatCallback = function(response) {
  if (response == 'UNKNOWN') {
    this.navigateToPath_(this.capturePath_);
    return;
  }
  var elapsed = this.getTime_() - this.sent_;
  this.sent_ = 0;

  this.view_.updateStatus(response);

  if (elapsed < this.interval_) {
    var interval = this.interval_ - (elapsed > 0 ? elapsed : 0);
    this.view_.updateNextBeat(interval);
    this.timeoutId_ = this.setTimeout_(this.boundSendHeartBeat_,
        interval);
  } else {
    this.sendHeartbeat();
  }
};



/**
 * Handles the rendering of the heartbeat monitor.
 * @param {function():HTMLBodyElement} Accessor for the body element of the page.
 * @param {function(String):HTMLElement} Element creator factory.
 */
jstestdriver.HeartbeatView = function(getBody, createElement) {
  this.getBody_ = getBody;
  this.createElement_ = createElement;
  this.root_ = null;
  this.status_ = null;
  this.lastBeat_ = null;
  this.nextBeat_ = null;
};


jstestdriver.HeartbeatView.prototype.ensureView = function() {
  if (!this.root_) {
    var body = this.getBody_();
    var document = body.ownerDocument;
    this.root_ = body.appendChild(document.createElement('p'));
    this.root_.innerHTML = 'JsTestDriver<br/>';
    this.lastBeat_ = this.root_.appendChild(document.createElement('span'));
    this.root_.appendChild(document.createTextNode(" | "));
    this.nextBeat_ = this.root_.appendChild(document.createElement('span'));
    this.root_.appendChild(document.createTextNode(" | "));
    this.status_ = this.root_.appendChild(document.createElement('span'));
  }
};


/**
 * Sets the status message for the last heartbeat.
 * @param {String}
 */
jstestdriver.HeartbeatView.prototype.updateLastBeat = function(msg) {
  this.ensureView();
  this.lastBeat_.innerHTML = ' Last:' + msg;
};


/**
 * Sets the status message for the next heartbeat.
 * @param {String}
 */
jstestdriver.HeartbeatView.prototype.updateNextBeat = function(msg) {
  this.ensureView();
  this.nextBeat_.innerHTML = ' Next:' + msg;
};


/**
 * Sets the status message for the heartbeat.
 * @param {String}
 */
jstestdriver.HeartbeatView.prototype.updateStatus = function(msg) {
  this.ensureView();
  this.status_.innerHTML = ' Server:' + msg;
};


/**
 * Renders the heartbeat connection status.
 * @param {boolean} The connection status.
 */
jstestdriver.HeartbeatView.prototype.updateConnected = function(connected) {
  if (!connected) {
    this.getBody_().className = jstestdriver.HeartbeatView.ERROR_CLASS;
    this.updateStatus('Lost server...');
  } else {
    this.getBody_().className = '';
    this.updateStatus('Waiting...');
  }
};


jstestdriver.HeartbeatView.ERROR_CLASS = 'error';
jstestdriver.createHeartbeat = function(capturePath) {
  function getBody() {
    return document.getElementsByTagName('body')[0];
  }

  function createElement(elementName) {
    return document.createElement(elementName);
  }

  var view = new jstestdriver.HeartbeatView(getBody, createElement);

  function sendRequest(url, data, success, error) {
    jstestdriver.jQuery.ajax({
      type : 'POST',
      url : url,
      data : data,
      success : success,
      error : error,
      dataType : 'text'
    });
  }

  function getTime() {
    return new Date().getTime();
  }

  function redirectToPath(path){
    top.location = path
  }

  var id = jstestdriver.extractId(window.location.toString());
  var heartbeatPath = jstestdriver.createPath(window.location.toString(),
      jstestdriver.HEARTBEAT_URL);
  return new jstestdriver.Heartbeat(id,
                                    heartbeatPath,
                                    capturePath,
                                    sendRequest,
                                    2000,
                                    jstestdriver.setTimeout,
                                    getTime,
                                    view,
                                    redirectToPath);
};
