/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile;

import java.io.ByteArrayOutputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataInputStream;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.MediumTests;
import org.apache.hadoop.hbase.fs.HFileSystem;
import org.apache.hadoop.hbase.io.compress.Compression;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheConfig;
import org.apache.hadoop.hbase.io.hfile.HFile;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.HFileBlockIndex;
import org.apache.hadoop.hbase.io.hfile.HFileContext;
import org.apache.hadoop.hbase.io.hfile.HFileContextBuilder;
import org.apache.hadoop.hbase.io.hfile.HFileReaderV2;
import org.apache.hadoop.hbase.io.hfile.HFileScanner;
import org.apache.hadoop.hbase.io.hfile.TestHFileWriterV2;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.ClassSize;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.experimental.categories.Category;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

@RunWith(value=Parameterized.class)
@Category(value={MediumTests.class})
public class TestHFileBlockIndex {
    private static final Log LOG = LogFactory.getLog(TestHFileBlockIndex.class);
    private static final int NUM_DATA_BLOCKS = 1000;
    private static final HBaseTestingUtility TEST_UTIL = new HBaseTestingUtility();
    private static final int SMALL_BLOCK_SIZE = 4096;
    private static final int NUM_KV = 10000;
    private static FileSystem fs;
    private Path path;
    private Random rand;
    private long rootIndexOffset;
    private int numRootEntries;
    private int numLevels;
    private static final List<byte[]> keys;
    private final Compression.Algorithm compr;
    private byte[] firstKeyInFile;
    private Configuration conf;
    private static final int[] INDEX_CHUNK_SIZES;
    private static final int[] EXPECTED_NUM_LEVELS;
    private static final int[] UNCOMPRESSED_INDEX_SIZES;
    private static final boolean includesMemstoreTS = true;

    @Parameterized.Parameters
    public static Collection<Object[]> compressionAlgorithms() {
        return HBaseTestingUtility.COMPRESSION_ALGORITHMS_PARAMETERIZED;
    }

    public TestHFileBlockIndex(Compression.Algorithm compr) {
        this.compr = compr;
    }

    @Before
    public void setUp() throws IOException {
        keys.clear();
        this.rand = new Random(2389757L);
        this.firstKeyInFile = null;
        this.conf = TEST_UTIL.getConfiguration();
        this.conf.setInt("hfile.format.version", 3);
        fs = HFileSystem.get(this.conf);
    }

    @Test
    public void testBlockIndex() throws IOException {
        this.testBlockIndexInternals(false);
        this.clear();
        this.testBlockIndexInternals(true);
    }

    private void clear() throws IOException {
        keys.clear();
        this.rand = new Random(2389757L);
        this.firstKeyInFile = null;
        this.conf = TEST_UTIL.getConfiguration();
        this.conf.setInt("hfile.format.version", 3);
        fs = HFileSystem.get(this.conf);
    }

    protected void testBlockIndexInternals(boolean useTags) throws IOException {
        this.path = new Path(TEST_UTIL.getDataTestDir(), "block_index_" + (Object)((Object)this.compr) + useTags);
        this.writeWholeIndex(useTags);
        this.readIndex(useTags);
    }

    public void readIndex(boolean useTags) throws IOException {
        long fileSize = fs.getFileStatus(this.path).getLen();
        LOG.info("Size of " + this.path + ": " + fileSize);
        FSDataInputStream istream = fs.open(this.path);
        HFileContext meta = new HFileContextBuilder().withHBaseCheckSum(true).withIncludesMvcc(true).withIncludesTags(useTags).withCompression(this.compr).build();
        HFileBlock.FSReaderV2 blockReader = new HFileBlock.FSReaderV2(istream, fs.getFileStatus(this.path).getLen(), meta);
        BlockReaderWrapper brw = new BlockReaderWrapper(blockReader);
        HFileBlockIndex.BlockIndexReader indexReader = new HFileBlockIndex.BlockIndexReader(KeyValue.RAW_COMPARATOR, this.numLevels, brw);
        indexReader.readRootIndex(blockReader.blockRange(this.rootIndexOffset, fileSize).nextBlockWithBlockType(BlockType.ROOT_INDEX), this.numRootEntries);
        long prevOffset = -1L;
        int i = 0;
        int expectedHitCount = 0;
        int expectedMissCount = 0;
        LOG.info("Total number of keys: " + keys.size());
        for (byte[] key : keys) {
            Assert.assertTrue((key != null ? 1 : 0) != 0);
            Assert.assertTrue((indexReader != null ? 1 : 0) != 0);
            HFileBlock b = indexReader.seekToDataBlock(key, 0, key.length, null, true, true, false);
            if (Bytes.BYTES_RAWCOMPARATOR.compare(key, this.firstKeyInFile) < 0) {
                Assert.assertTrue((b == null ? 1 : 0) != 0);
                ++i;
                continue;
            }
            String keyStr = "key #" + i + ", " + Bytes.toStringBinary(key);
            Assert.assertTrue((String)("seekToDataBlock failed for " + keyStr), (b != null ? 1 : 0) != 0);
            if (prevOffset == b.getOffset()) {
                Assert.assertEquals((long)(++expectedHitCount), (long)brw.hitCount);
            } else {
                LOG.info("First key in a new block: " + keyStr + ", block offset: " + b.getOffset() + ")");
                Assert.assertTrue((b.getOffset() > prevOffset ? 1 : 0) != 0);
                Assert.assertEquals((long)(++expectedMissCount), (long)brw.missCount);
                prevOffset = b.getOffset();
            }
            ++i;
        }
        istream.close();
    }

    private void writeWholeIndex(boolean useTags) throws IOException {
        Assert.assertEquals((long)0L, (long)keys.size());
        HFileContext meta = new HFileContextBuilder().withHBaseCheckSum(true).withIncludesMvcc(true).withIncludesTags(useTags).withCompression(this.compr).withChecksumType(HFile.DEFAULT_CHECKSUM_TYPE).withBytesPerCheckSum(16384).build();
        HFileBlock.Writer hbw = new HFileBlock.Writer(null, meta);
        FSDataOutputStream outputStream = fs.create(this.path);
        HFileBlockIndex.BlockIndexWriter biw = new HFileBlockIndex.BlockIndexWriter(hbw, null, null);
        for (int i = 0; i < 1000; ++i) {
            hbw.startWriting(BlockType.DATA).write(String.valueOf(this.rand.nextInt(1000)).getBytes());
            long blockOffset = outputStream.getPos();
            hbw.writeHeaderAndData(outputStream);
            byte[] firstKey = null;
            for (int j = 0; j < 16; ++j) {
                byte[] k = TestHFileWriterV2.randomOrderedKey(this.rand, i * 16 + j);
                keys.add(k);
                if (j != 8) continue;
                firstKey = k;
            }
            Assert.assertTrue((firstKey != null ? 1 : 0) != 0);
            if (this.firstKeyInFile == null) {
                this.firstKeyInFile = firstKey;
            }
            biw.addEntry(firstKey, blockOffset, hbw.getOnDiskSizeWithHeader());
            this.writeInlineBlocks(hbw, outputStream, biw, false);
        }
        this.writeInlineBlocks(hbw, outputStream, biw, true);
        this.rootIndexOffset = biw.writeIndexBlocks(outputStream);
        outputStream.close();
        this.numLevels = biw.getNumLevels();
        this.numRootEntries = biw.getNumRootEntries();
        LOG.info("Index written: numLevels=" + this.numLevels + ", numRootEntries=" + this.numRootEntries + ", rootIndexOffset=" + this.rootIndexOffset);
    }

    private void writeInlineBlocks(HFileBlock.Writer hbw, FSDataOutputStream outputStream, HFileBlockIndex.BlockIndexWriter biw, boolean isClosing) throws IOException {
        while (biw.shouldWriteBlock(isClosing)) {
            long offset = outputStream.getPos();
            biw.writeInlineBlock(hbw.startWriting(biw.getInlineBlockType()));
            hbw.writeHeaderAndData(outputStream);
            biw.blockWritten(offset, hbw.getOnDiskSizeWithHeader(), hbw.getUncompressedSizeWithoutHeader());
            LOG.info("Wrote an inline index block at " + offset + ", size " + hbw.getOnDiskSizeWithHeader());
        }
    }

    private static final long getDummyFileOffset(int i) {
        return i * 185 + 379;
    }

    private static final int getDummyOnDiskSize(int i) {
        return i * i * 37 + i * 19 + 13;
    }

    @Test
    public void testSecondaryIndexBinarySearch() throws IOException {
        int i;
        int numTotalKeys = 99;
        Assert.assertTrue((numTotalKeys % 2 == 1 ? 1 : 0) != 0);
        int numSearchedKeys = (numTotalKeys - 1) / 2;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(baos);
        dos.writeInt(numSearchedKeys);
        int curAllEntriesSize = 0;
        int numEntriesAdded = 0;
        int[] secondaryIndexEntries = new int[numTotalKeys];
        for (i = 0; i < numTotalKeys; ++i) {
            byte[] k = TestHFileWriterV2.randomOrderedKey(this.rand, i * 2);
            keys.add(k);
            String msgPrefix = "Key #" + i + " (" + Bytes.toStringBinary(k) + "): ";
            StringBuilder padding = new StringBuilder();
            while (msgPrefix.length() + padding.length() < 70) {
                padding.append(' ');
            }
            msgPrefix = msgPrefix + padding;
            if (i % 2 == 1) {
                dos.writeInt(curAllEntriesSize);
                secondaryIndexEntries[i] = curAllEntriesSize;
                LOG.info(msgPrefix + "secondary index entry #" + (i - 1) / 2 + ", offset " + curAllEntriesSize);
                curAllEntriesSize += k.length + 12;
                ++numEntriesAdded;
                continue;
            }
            secondaryIndexEntries[i] = -1;
            LOG.info(msgPrefix + "not in the searched array");
        }
        for (i = 0; i < keys.size() - 1; ++i) {
            Assert.assertTrue((Bytes.BYTES_RAWCOMPARATOR.compare(keys.get(i), keys.get(i + 1)) < 0 ? 1 : 0) != 0);
        }
        dos.writeInt(curAllEntriesSize);
        Assert.assertEquals((long)numSearchedKeys, (long)numEntriesAdded);
        int secondaryIndexOffset = dos.size();
        Assert.assertEquals((long)(4 * (numSearchedKeys + 2)), (long)secondaryIndexOffset);
        for (int i2 = 1; i2 <= numTotalKeys - 1; i2 += 2) {
            Assert.assertEquals((long)dos.size(), (long)(secondaryIndexOffset + secondaryIndexEntries[i2]));
            long dummyFileOffset = TestHFileBlockIndex.getDummyFileOffset(i2);
            int dummyOnDiskSize = TestHFileBlockIndex.getDummyOnDiskSize(i2);
            LOG.debug("Storing file offset=" + dummyFileOffset + " and onDiskSize=" + dummyOnDiskSize + " at offset " + dos.size());
            dos.writeLong(dummyFileOffset);
            dos.writeInt(dummyOnDiskSize);
            LOG.debug("Stored key " + (i2 - 1) / 2 + " at offset " + dos.size());
            dos.write(keys.get(i2));
        }
        dos.writeInt(curAllEntriesSize);
        ByteBuffer nonRootIndex = ByteBuffer.wrap(baos.toByteArray());
        for (int i3 = 0; i3 < numTotalKeys; ++i3) {
            boolean locateBlockResult;
            int referenceItem;
            int expectedResult;
            byte[] searchKey = keys.get(i3);
            byte[] arrayHoldingKey = new byte[searchKey.length + searchKey.length / 2];
            System.arraycopy(searchKey, 0, arrayHoldingKey, searchKey.length / 2, searchKey.length);
            int searchResult = HFileBlockIndex.BlockIndexReader.binarySearchNonRootIndex(arrayHoldingKey, searchKey.length / 2, searchKey.length, nonRootIndex, KeyValue.RAW_COMPARATOR);
            String lookupFailureMsg = "Failed to look up key #" + i3 + " (" + Bytes.toStringBinary(searchKey) + ")";
            if (i3 % 2 == 1) {
                expectedResult = (i3 - 1) / 2;
                referenceItem = i3;
            } else {
                expectedResult = i3 / 2 - 1;
                referenceItem = i3 - 1;
            }
            Assert.assertEquals((String)lookupFailureMsg, (long)expectedResult, (long)searchResult);
            boolean bl = locateBlockResult = HFileBlockIndex.BlockIndexReader.locateNonRootIndexEntry(nonRootIndex, arrayHoldingKey, searchKey.length / 2, searchKey.length, KeyValue.RAW_COMPARATOR) != -1;
            if (i3 == 0) {
                Assert.assertFalse((boolean)locateBlockResult);
                continue;
            }
            Assert.assertTrue((boolean)locateBlockResult);
            String errorMsg = "i=" + i3 + ", position=" + nonRootIndex.position();
            Assert.assertEquals((String)errorMsg, (long)TestHFileBlockIndex.getDummyFileOffset(referenceItem), (long)nonRootIndex.getLong());
            Assert.assertEquals((String)errorMsg, (long)TestHFileBlockIndex.getDummyOnDiskSize(referenceItem), (long)nonRootIndex.getInt());
        }
    }

    @Test
    public void testBlockIndexChunk() throws IOException {
        int i;
        HFileBlockIndex.BlockIndexChunk c = new HFileBlockIndex.BlockIndexChunk();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        int N = 1000;
        int[] numSubEntriesAt = new int[N];
        int numSubEntries = 0;
        for (i = 0; i < N; ++i) {
            baos.reset();
            DataOutputStream dos = new DataOutputStream(baos);
            c.writeNonRoot(dos);
            Assert.assertEquals((long)c.getNonRootSize(), (long)dos.size());
            baos.reset();
            dos = new DataOutputStream(baos);
            c.writeRoot(dos);
            Assert.assertEquals((long)c.getRootSize(), (long)dos.size());
            byte[] k = TestHFileWriterV2.randomOrderedKey(this.rand, i);
            keys.add(k);
            c.add(k, TestHFileBlockIndex.getDummyFileOffset(i), TestHFileBlockIndex.getDummyOnDiskSize(i), numSubEntries += this.rand.nextInt(5) + 1);
        }
        for (i = 0; i < N; ++i) {
            int j;
            int n = j = i == 0 ? 0 : numSubEntriesAt[i - 1];
            while (j < numSubEntriesAt[i]) {
                Assert.assertEquals((long)i, (long)c.getEntryBySubEntry(j));
                ++j;
            }
        }
    }

    @Test
    public void testHeapSizeForBlockIndex() throws IOException {
        Class<HFileBlockIndex.BlockIndexReader> cl = HFileBlockIndex.BlockIndexReader.class;
        long expected = ClassSize.estimateBase(cl, false);
        HFileBlockIndex.BlockIndexReader bi = new HFileBlockIndex.BlockIndexReader(KeyValue.RAW_COMPARATOR, 1);
        long actual = bi.heapSize();
        if ((expected -= (long)ClassSize.align(3 * ClassSize.ARRAY)) != actual) {
            ClassSize.estimateBase(cl, true);
            Assert.assertEquals((long)expected, (long)actual);
        }
    }

    @Test
    public void testHFileWriterAndReader() throws IOException {
        Path hfilePath = new Path(TEST_UTIL.getDataTestDir(), "hfile_for_block_index");
        CacheConfig cacheConf = new CacheConfig(this.conf);
        BlockCache blockCache = cacheConf.getBlockCache();
        for (int testI = 0; testI < INDEX_CHUNK_SIZES.length; ++testI) {
            HFileBlock block;
            int indexBlockSize = INDEX_CHUNK_SIZES[testI];
            int expectedNumLevels = EXPECTED_NUM_LEVELS[testI];
            LOG.info("Index block size: " + indexBlockSize + ", compression: " + (Object)((Object)this.compr));
            blockCache.evictBlocksByHfileName(hfilePath.getName());
            this.conf.setInt("hfile.index.block.max.size", indexBlockSize);
            HashSet<String> keyStrSet = new HashSet<String>();
            byte[][] keys = new byte[10000][];
            byte[][] values = new byte[10000][];
            HFileContext meta = new HFileContextBuilder().withBlockSize(4096).withCompression(this.compr).build();
            HFile.Writer writer = HFile.getWriterFactory(this.conf, cacheConf).withPath(fs, hfilePath).withFileContext(meta).create();
            Random rand = new Random(19231737L);
            for (int i = 0; i < 10000; ++i) {
                byte[] row = TestHFileWriterV2.randomOrderedKey(rand, i);
                byte[] k = KeyValue.createFirstOnRow(row, 0, row.length, row, 0, 0, row, 0, 0).getKey();
                byte[] v = TestHFileWriterV2.randomValue(rand);
                writer.append(k, v, HConstants.EMPTY_BYTE_ARRAY);
                keys[i] = k;
                values[i] = v;
                keyStrSet.add(Bytes.toStringBinary(k));
                if (i <= 0) continue;
                Assert.assertTrue((KeyValue.COMPARATOR.compareFlatKey(keys[i - 1], keys[i]) < 0 ? 1 : 0) != 0);
            }
            writer.close();
            HFile.Reader reader = HFile.createReader(fs, hfilePath, cacheConf, this.conf);
            Assert.assertEquals((long)expectedNumLevels, (long)reader.getTrailer().getNumDataIndexLevels());
            Assert.assertTrue((boolean)Bytes.equals(keys[0], reader.getFirstKey()));
            Assert.assertTrue((boolean)Bytes.equals(keys[9999], reader.getLastKey()));
            LOG.info("Last key: " + Bytes.toStringBinary(keys[9999]));
            for (boolean pread : new boolean[]{false, true}) {
                int i;
                HFileScanner scanner = reader.getScanner(true, pread);
                for (i = 0; i < 10000; ++i) {
                    this.checkSeekTo(keys, scanner, i);
                    this.checkKeyValue("i=" + i, keys[i], values[i], scanner.getKey(), scanner.getValue());
                }
                Assert.assertTrue((boolean)scanner.seekTo());
                for (i = 9999; i >= 0; --i) {
                    this.checkSeekTo(keys, scanner, i);
                    this.checkKeyValue("i=" + i, keys[i], values[i], scanner.getKey(), scanner.getValue());
                }
            }
            HFileReaderV2 reader2 = (HFileReaderV2)reader;
            HFileBlock.FSReader fsReader = reader2.getUncachedBlockReader();
            HFileBlock.BlockIterator iter = fsReader.blockRange(0L, reader.getTrailer().getLoadOnOpenDataOffset());
            ArrayList<byte[]> blockKeys = new ArrayList<byte[]>();
            while ((block = iter.nextBlock()) != null) {
                if (block.getBlockType() != BlockType.LEAF_INDEX) {
                    return;
                }
                ByteBuffer b = block.getBufferReadOnly();
                int n = b.getInt();
                int entriesOffset = 4 * (n + 2);
                for (int i = 0; i < n; ++i) {
                    int keyRelOffset = b.getInt(4 * (i + 1));
                    int nextKeyRelOffset = b.getInt(4 * (i + 2));
                    int keyLen = nextKeyRelOffset - keyRelOffset;
                    int keyOffset = b.arrayOffset() + entriesOffset + keyRelOffset + 12;
                    byte[] blockKey = Arrays.copyOfRange(b.array(), keyOffset, keyOffset + keyLen);
                    String blockKeyStr = Bytes.toString(blockKey);
                    blockKeys.add(blockKey);
                    Assert.assertTrue((String)("Invalid block key from leaf-level block: " + blockKeyStr), (boolean)keyStrSet.contains(blockKeyStr));
                }
            }
            Assert.assertEquals((Object)Bytes.toStringBinary((byte[])blockKeys.get((blockKeys.size() - 1) / 2)), (Object)Bytes.toStringBinary(reader.midkey()));
            Assert.assertEquals((long)UNCOMPRESSED_INDEX_SIZES[testI], (long)reader.getTrailer().getUncompressedDataIndexSize());
            reader.close();
            reader2.close();
        }
    }

    private void checkSeekTo(byte[][] keys, HFileScanner scanner, int i) throws IOException {
        Assert.assertEquals((String)("Failed to seek to key #" + i + " (" + Bytes.toStringBinary(keys[i]) + ")"), (long)0L, (long)scanner.seekTo(keys[i]));
    }

    private void assertArrayEqualsBuffer(String msgPrefix, byte[] arr, ByteBuffer buf) {
        Assert.assertEquals((String)(msgPrefix + ": expected " + Bytes.toStringBinary(arr) + ", actual " + Bytes.toStringBinary(buf)), (long)0L, (long)Bytes.compareTo(arr, 0, arr.length, buf.array(), buf.arrayOffset(), buf.limit()));
    }

    private void checkKeyValue(String msgPrefix, byte[] expectedKey, byte[] expectedValue, ByteBuffer keyRead, ByteBuffer valueRead) {
        if (!msgPrefix.isEmpty()) {
            msgPrefix = msgPrefix + ". ";
        }
        this.assertArrayEqualsBuffer(msgPrefix + "Invalid key", expectedKey, keyRead);
        this.assertArrayEqualsBuffer(msgPrefix + "Invalid value", expectedValue, valueRead);
    }

    static {
        keys = new ArrayList<byte[]>();
        INDEX_CHUNK_SIZES = new int[]{4096, 512, 384};
        EXPECTED_NUM_LEVELS = new int[]{2, 3, 4};
        UNCOMPRESSED_INDEX_SIZES = new int[]{19187, 21813, 23086};
        assert (INDEX_CHUNK_SIZES.length == EXPECTED_NUM_LEVELS.length);
        assert (INDEX_CHUNK_SIZES.length == UNCOMPRESSED_INDEX_SIZES.length);
    }

    private static class BlockReaderWrapper
    implements HFile.CachingBlockReader {
        private HFileBlock.FSReader realReader;
        private long prevOffset;
        private long prevOnDiskSize;
        private boolean prevPread;
        private HFileBlock prevBlock;
        public int hitCount = 0;
        public int missCount = 0;

        public BlockReaderWrapper(HFileBlock.FSReader realReader) {
            this.realReader = realReader;
        }

        @Override
        public HFileBlock readBlock(long offset, long onDiskSize, boolean cacheBlock, boolean pread, boolean isCompaction, BlockType expectedBlockType) throws IOException {
            if (offset == this.prevOffset && onDiskSize == this.prevOnDiskSize && pread == this.prevPread) {
                ++this.hitCount;
                return this.prevBlock;
            }
            ++this.missCount;
            this.prevBlock = this.realReader.readBlockData(offset, onDiskSize, -1, pread);
            this.prevOffset = offset;
            this.prevOnDiskSize = onDiskSize;
            this.prevPread = pread;
            return this.prevBlock;
        }
    }
}

