/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import com.google.common.annotations.VisibleForTesting;
import com.google.protobuf.ServiceException;
import java.io.IOException;
import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedMap;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentSkipListMap;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.ClockOutOfSyncException;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.RegionLoad;
import org.apache.hadoop.hbase.Server;
import org.apache.hadoop.hbase.ServerLoad;
import org.apache.hadoop.hbase.ServerName;
import org.apache.hadoop.hbase.YouAreDeadException;
import org.apache.hadoop.hbase.client.HConnection;
import org.apache.hadoop.hbase.client.HConnectionManager;
import org.apache.hadoop.hbase.master.DeadServer;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.handler.MetaServerShutdownHandler;
import org.apache.hadoop.hbase.master.handler.ServerShutdownHandler;
import org.apache.hadoop.hbase.monitoring.MonitoredTask;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.protobuf.RequestConverter;
import org.apache.hadoop.hbase.protobuf.ResponseConverter;
import org.apache.hadoop.hbase.protobuf.generated.AdminProtos;
import org.apache.hadoop.hbase.regionserver.RegionOpeningState;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.Triple;

@InterfaceAudience.Private
public class ServerManager {
    public static final String WAIT_ON_REGIONSERVERS_MAXTOSTART = "hbase.master.wait.on.regionservers.maxtostart";
    public static final String WAIT_ON_REGIONSERVERS_MINTOSTART = "hbase.master.wait.on.regionservers.mintostart";
    public static final String WAIT_ON_REGIONSERVERS_TIMEOUT = "hbase.master.wait.on.regionservers.timeout";
    public static final String WAIT_ON_REGIONSERVERS_INTERVAL = "hbase.master.wait.on.regionservers.interval";
    private static final Log LOG = LogFactory.getLog(ServerManager.class);
    private volatile boolean clusterShutdown = false;
    private final SortedMap<byte[], Long> flushedSequenceIdByRegion = new ConcurrentSkipListMap<byte[], Long>(Bytes.BYTES_COMPARATOR);
    private final ConcurrentHashMap<ServerName, ServerLoad> onlineServers = new ConcurrentHashMap();
    private final Map<ServerName, AdminProtos.AdminService.BlockingInterface> rsAdmins = new HashMap<ServerName, AdminProtos.AdminService.BlockingInterface>();
    private final ArrayList<ServerName> drainingServers = new ArrayList();
    private final Server master;
    private final MasterServices services;
    private final HConnection connection;
    private final DeadServer deadservers = new DeadServer();
    private final long maxSkew;
    private final long warningSkew;
    private Set<ServerName> queuedDeadServers = new HashSet<ServerName>();
    private Map<ServerName, Boolean> requeuedDeadServers = new HashMap<ServerName, Boolean>();

    public ServerManager(Server master, MasterServices services) throws IOException {
        this(master, services, true);
    }

    ServerManager(Server master, MasterServices services, boolean connect) throws IOException {
        this.master = master;
        this.services = services;
        Configuration c = master.getConfiguration();
        this.maxSkew = c.getLong("hbase.master.maxclockskew", 30000L);
        this.warningSkew = c.getLong("hbase.master.warningclockskew", 10000L);
        this.connection = connect ? HConnectionManager.getConnection(c) : null;
    }

    ServerName regionServerStartup(InetAddress ia, int port, long serverStartcode, long serverCurrentTime) throws IOException {
        ServerName sn = ServerName.valueOf(ia.getHostName(), port, serverStartcode);
        this.checkClockSkew(sn, serverCurrentTime);
        this.checkIsDead(sn, "STARTUP");
        if (!this.checkAndRecordNewServer(sn, ServerLoad.EMPTY_SERVERLOAD)) {
            LOG.warn("THIS SHOULD NOT HAPPEN, RegionServerStartup could not record the server: " + sn);
        }
        return sn;
    }

    private void updateLastFlushedSequenceIds(ServerName sn, ServerLoad hsl) {
        Map<byte[], RegionLoad> regionsLoad = hsl.getRegionsLoad();
        for (Map.Entry<byte[], RegionLoad> entry : regionsLoad.entrySet()) {
            Long existingValue = (Long)this.flushedSequenceIdByRegion.get(entry.getKey());
            long l = entry.getValue().getCompleteSequenceId();
            if (existingValue != null && l != -1L && l < existingValue) {
                if (!LOG.isDebugEnabled()) continue;
                LOG.debug("RegionServer " + sn + " indicates a last flushed sequence id (" + entry.getValue() + ") that is less than the previous last flushed sequence id (" + existingValue + ") for region " + Bytes.toString(entry.getKey()) + " Ignoring.");
                continue;
            }
            this.flushedSequenceIdByRegion.put(entry.getKey(), l);
        }
    }

    void regionServerReport(ServerName sn, ServerLoad sl) throws YouAreDeadException {
        this.checkIsDead(sn, "REPORT");
        if (null == this.onlineServers.replace(sn, sl) && !this.checkAndRecordNewServer(sn, sl)) {
            LOG.info("RegionServerReport ignored, could not record the server: " + sn);
            return;
        }
        this.updateLastFlushedSequenceIds(sn, sl);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    boolean checkAndRecordNewServer(ServerName serverName, ServerLoad sl) {
        ServerName existingServer = null;
        ConcurrentHashMap<ServerName, ServerLoad> concurrentHashMap = this.onlineServers;
        synchronized (concurrentHashMap) {
            existingServer = this.findServerWithSameHostnamePortWithLock(serverName);
            if (existingServer != null && existingServer.getStartcode() > serverName.getStartcode()) {
                LOG.info("Server serverName=" + serverName + " rejected; we already have " + existingServer.toString() + " registered with same hostname and port");
                return false;
            }
            this.recordNewServerWithLock(serverName, sl);
        }
        if (existingServer != null && existingServer.getStartcode() < serverName.getStartcode()) {
            LOG.info("Triggering server recovery; existingServer " + existingServer + " looks stale, new server:" + serverName);
            this.expireServer(existingServer);
        }
        return true;
    }

    private void checkClockSkew(ServerName serverName, long serverCurrentTime) throws ClockOutOfSyncException {
        long skew = Math.abs(System.currentTimeMillis() - serverCurrentTime);
        if (skew > this.maxSkew) {
            String message = "Server " + serverName + " has been " + "rejected; Reported time is too far out of sync with master.  " + "Time difference of " + skew + "ms > max allowed of " + this.maxSkew + "ms";
            LOG.warn(message);
            throw new ClockOutOfSyncException(message);
        }
        if (skew > this.warningSkew) {
            String message = "Reported time for server " + serverName + " is out of sync with master " + "by " + skew + "ms. (Warning threshold is " + this.warningSkew + "ms; " + "error threshold is " + this.maxSkew + "ms)";
            LOG.warn(message);
        }
    }

    private void checkIsDead(ServerName serverName, String what) throws YouAreDeadException {
        if (this.deadservers.isDeadServer(serverName)) {
            String message = "Server " + what + " rejected; currently processing " + serverName + " as dead server";
            LOG.debug(message);
            throw new YouAreDeadException(message);
        }
        if ((this.services == null || ((HMaster)this.services).isInitialized()) && this.deadservers.cleanPreviousInstance(serverName)) {
            LOG.debug(what + ":" + " Server " + serverName + " came back up," + " removed it from the dead servers list");
        }
    }

    private ServerName findServerWithSameHostnamePortWithLock(ServerName serverName) {
        for (ServerName sn : this.onlineServers.keySet()) {
            if (!ServerName.isSameHostnameAndPort(serverName, sn)) continue;
            return sn;
        }
        return null;
    }

    @VisibleForTesting
    void recordNewServerWithLock(ServerName serverName, ServerLoad sl) {
        LOG.info("Registering server=" + serverName);
        this.onlineServers.put(serverName, sl);
        this.rsAdmins.remove(serverName);
    }

    public long getLastFlushedSequenceId(byte[] regionName) {
        long seqId = -1L;
        if (this.flushedSequenceIdByRegion.containsKey(regionName)) {
            seqId = (Long)this.flushedSequenceIdByRegion.get(regionName);
        }
        return seqId;
    }

    public ServerLoad getLoad(ServerName serverName) {
        return this.onlineServers.get(serverName);
    }

    public double getAverageLoad() {
        int totalLoad = 0;
        int numServers = 0;
        for (ServerLoad sl : this.onlineServers.values()) {
            ++numServers;
            totalLoad += sl.getNumberOfRegions();
        }
        double averageLoad = (double)totalLoad / (double)numServers;
        return averageLoad;
    }

    int countOfRegionServers() {
        return this.onlineServers.size();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public Map<ServerName, ServerLoad> getOnlineServers() {
        ConcurrentHashMap<ServerName, ServerLoad> concurrentHashMap = this.onlineServers;
        synchronized (concurrentHashMap) {
            return Collections.unmodifiableMap(this.onlineServers);
        }
    }

    public DeadServer getDeadServers() {
        return this.deadservers;
    }

    public boolean areDeadServersInProgress() {
        return this.deadservers.areDeadServersInProgress();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void letRegionServersShutdown() {
        long previousLogTime = 0L;
        while (!this.onlineServers.isEmpty()) {
            if (System.currentTimeMillis() > previousLogTime + 1000L) {
                StringBuilder sb = new StringBuilder();
                for (ServerName key : this.onlineServers.keySet()) {
                    if (sb.length() > 0) {
                        sb.append(", ");
                    }
                    sb.append(key);
                }
                LOG.info("Waiting on regionserver(s) to go down " + sb.toString());
                previousLogTime = System.currentTimeMillis();
            }
            ConcurrentHashMap<ServerName, ServerLoad> concurrentHashMap = this.onlineServers;
            synchronized (concurrentHashMap) {
                try {
                    this.onlineServers.wait(100L);
                }
                catch (InterruptedException ignored) {
                    // empty catch block
                }
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized void expireServer(ServerName serverName) {
        if (!this.services.isServerShutdownHandlerEnabled()) {
            LOG.info("Master doesn't enable ServerShutdownHandler during initialization, delay expiring server " + serverName);
            this.queuedDeadServers.add(serverName);
            return;
        }
        if (this.deadservers.isDeadServer(serverName)) {
            LOG.warn("Expiration of " + serverName + " but server shutdown already in progress");
            return;
        }
        ConcurrentHashMap<ServerName, ServerLoad> concurrentHashMap = this.onlineServers;
        synchronized (concurrentHashMap) {
            if (!this.onlineServers.containsKey(serverName)) {
                LOG.warn("Expiration of " + serverName + " but server not online");
            }
            this.deadservers.add(serverName);
            this.onlineServers.remove(serverName);
            this.onlineServers.notifyAll();
        }
        this.rsAdmins.remove(serverName);
        if (this.clusterShutdown) {
            LOG.info("Cluster shutdown set; " + serverName + " expired; onlineServers=" + this.onlineServers.size());
            if (this.onlineServers.isEmpty()) {
                this.master.stop("Cluster shutdown set; onlineServer=0");
            }
            return;
        }
        boolean carryingMeta = this.services.getAssignmentManager().isCarryingMeta(serverName);
        if (carryingMeta) {
            this.services.getExecutorService().submit(new MetaServerShutdownHandler(this.master, this.services, this.deadservers, serverName));
        } else {
            this.services.getExecutorService().submit(new ServerShutdownHandler(this.master, this.services, this.deadservers, serverName, true));
        }
        LOG.debug("Added=" + serverName + " to dead servers, submitted shutdown handler to be executed meta=" + carryingMeta);
    }

    public synchronized void processDeadServer(ServerName serverName) {
        this.processDeadServer(serverName, false);
    }

    public synchronized void processDeadServer(ServerName serverName, boolean shouldSplitHlog) {
        if (!this.services.getAssignmentManager().isFailoverCleanupDone()) {
            this.requeuedDeadServers.put(serverName, shouldSplitHlog);
            return;
        }
        this.deadservers.add(serverName);
        this.services.getExecutorService().submit(new ServerShutdownHandler(this.master, this.services, this.deadservers, serverName, shouldSplitHlog));
    }

    synchronized void processQueuedDeadServers() {
        if (!this.services.isServerShutdownHandlerEnabled()) {
            LOG.info("Master hasn't enabled ServerShutdownHandler");
        }
        Iterator<ServerName> serverIterator = this.queuedDeadServers.iterator();
        while (serverIterator.hasNext()) {
            ServerName tmpServerName = serverIterator.next();
            this.expireServer(tmpServerName);
            serverIterator.remove();
            this.requeuedDeadServers.remove(tmpServerName);
        }
        if (!this.services.getAssignmentManager().isFailoverCleanupDone()) {
            LOG.info("AssignmentManager hasn't finished failover cleanup; waiting");
        }
        for (ServerName tmpServerName : this.requeuedDeadServers.keySet()) {
            this.processDeadServer(tmpServerName, this.requeuedDeadServers.get(tmpServerName));
        }
        this.requeuedDeadServers.clear();
    }

    public boolean removeServerFromDrainList(ServerName sn) {
        if (!this.isServerOnline(sn)) {
            LOG.warn("Server " + sn + " is not currently online. " + "Removing from draining list anyway, as requested.");
        }
        return this.drainingServers.remove(sn);
    }

    public boolean addServerToDrainList(ServerName sn) {
        if (!this.isServerOnline(sn)) {
            LOG.warn("Server " + sn + " is not currently online. " + "Ignoring request to add it to draining list.");
            return false;
        }
        if (this.drainingServers.contains(sn)) {
            LOG.warn("Server " + sn + " is already in the draining server list." + "Ignoring request to add it again.");
            return false;
        }
        return this.drainingServers.add(sn);
    }

    public RegionOpeningState sendRegionOpen(ServerName server, HRegionInfo region, int versionOfOfflineNode, List<ServerName> favoredNodes) throws IOException {
        AdminProtos.AdminService.BlockingInterface admin = this.getRsAdmin(server);
        if (admin == null) {
            LOG.warn("Attempting to send OPEN RPC to server " + server.toString() + " failed because no RPC connection found to this server");
            return RegionOpeningState.FAILED_OPENING;
        }
        AdminProtos.OpenRegionRequest request = RequestConverter.buildOpenRegionRequest(server, region, versionOfOfflineNode, favoredNodes);
        try {
            AdminProtos.OpenRegionResponse response = admin.openRegion(null, request);
            return ResponseConverter.getRegionOpeningState(response);
        }
        catch (ServiceException se) {
            throw ProtobufUtil.getRemoteException(se);
        }
    }

    public List<RegionOpeningState> sendRegionOpen(ServerName server, List<Triple<HRegionInfo, Integer, List<ServerName>>> regionOpenInfos) throws IOException {
        AdminProtos.AdminService.BlockingInterface admin = this.getRsAdmin(server);
        if (admin == null) {
            LOG.warn("Attempting to send OPEN RPC to server " + server.toString() + " failed because no RPC connection found to this server");
            return null;
        }
        AdminProtos.OpenRegionRequest request = RequestConverter.buildOpenRegionRequest(regionOpenInfos);
        try {
            AdminProtos.OpenRegionResponse response = admin.openRegion(null, request);
            return ResponseConverter.getRegionOpeningStateList(response);
        }
        catch (ServiceException se) {
            throw ProtobufUtil.getRemoteException(se);
        }
    }

    public boolean sendRegionClose(ServerName server, HRegionInfo region, int versionOfClosingNode, ServerName dest, boolean transitionInZK) throws IOException {
        if (server == null) {
            throw new NullPointerException("Passed server is null");
        }
        AdminProtos.AdminService.BlockingInterface admin = this.getRsAdmin(server);
        if (admin == null) {
            throw new IOException("Attempting to send CLOSE RPC to server " + server.toString() + " for region " + region.getRegionNameAsString() + " failed because no RPC connection found to this server");
        }
        return ProtobufUtil.closeRegion(admin, server, region.getRegionName(), versionOfClosingNode, dest, transitionInZK);
    }

    public boolean sendRegionClose(ServerName server, HRegionInfo region, int versionOfClosingNode) throws IOException {
        return this.sendRegionClose(server, region, versionOfClosingNode, null, true);
    }

    public void sendRegionsMerge(ServerName server, HRegionInfo region_a, HRegionInfo region_b, boolean forcible) throws IOException {
        if (server == null) {
            throw new NullPointerException("Passed server is null");
        }
        if (region_a == null || region_b == null) {
            throw new NullPointerException("Passed region is null");
        }
        AdminProtos.AdminService.BlockingInterface admin = this.getRsAdmin(server);
        if (admin == null) {
            throw new IOException("Attempting to send MERGE REGIONS RPC to server " + server.toString() + " for region " + region_a.getRegionNameAsString() + "," + region_b.getRegionNameAsString() + " failed because no RPC connection found to this server");
        }
        ProtobufUtil.mergeRegions(admin, region_a, region_b, forcible);
    }

    public boolean isServerReachable(ServerName server) {
        if (server == null) {
            throw new NullPointerException("Passed server is null");
        }
        int maximumAttempts = Math.max(1, this.master.getConfiguration().getInt("hbase.master.maximum.ping.server.attempts", 10));
        for (int i = 0; i < maximumAttempts; ++i) {
            try {
                AdminProtos.AdminService.BlockingInterface admin = this.getRsAdmin(server);
                if (admin == null) continue;
                AdminProtos.ServerInfo info = ProtobufUtil.getServerInfo(admin);
                return info != null && info.hasServerName() && server.getStartcode() == info.getServerName().getStartCode();
            }
            catch (IOException ioe) {
                LOG.debug("Couldn't reach " + server + ", try=" + i + " of " + maximumAttempts, ioe);
            }
        }
        return false;
    }

    private AdminProtos.AdminService.BlockingInterface getRsAdmin(ServerName sn) throws IOException {
        AdminProtos.AdminService.BlockingInterface admin = this.rsAdmins.get(sn);
        if (admin == null) {
            LOG.debug("New admin connection to " + sn.toString());
            admin = this.connection.getAdmin(sn);
            this.rsAdmins.put(sn, admin);
        }
        return admin;
    }

    public void waitForRegionServers(MonitoredTask status) throws InterruptedException {
        long now;
        int maxToStart;
        long interval = this.master.getConfiguration().getLong(WAIT_ON_REGIONSERVERS_INTERVAL, 1500L);
        long timeout = this.master.getConfiguration().getLong(WAIT_ON_REGIONSERVERS_TIMEOUT, 4500L);
        int minToStart = this.master.getConfiguration().getInt(WAIT_ON_REGIONSERVERS_MINTOSTART, 1);
        if (minToStart < 1) {
            LOG.warn(String.format("The value of '%s' (%d) can not be less than 1, ignoring.", WAIT_ON_REGIONSERVERS_MINTOSTART, minToStart));
            minToStart = 1;
        }
        if ((maxToStart = this.master.getConfiguration().getInt(WAIT_ON_REGIONSERVERS_MAXTOSTART, Integer.MAX_VALUE)) < minToStart) {
            LOG.warn(String.format("The value of '%s' (%d) is set less than '%s' (%d), ignoring.", WAIT_ON_REGIONSERVERS_MAXTOSTART, maxToStart, WAIT_ON_REGIONSERVERS_MINTOSTART, minToStart));
            maxToStart = Integer.MAX_VALUE;
        }
        long startTime = now = System.currentTimeMillis();
        long slept = 0L;
        long lastLogTime = 0L;
        long lastCountChange = startTime;
        int count = this.countOfRegionServers();
        int oldCount = 0;
        while (!(this.master.isStopped() || count >= maxToStart || lastCountChange + interval <= now && timeout <= slept && count >= minToStart)) {
            if (oldCount != count || lastLogTime + interval < now) {
                lastLogTime = now;
                String msg = "Waiting for region servers count to settle; currently checked in " + count + ", slept for " + slept + " ms," + " expecting minimum of " + minToStart + ", maximum of " + maxToStart + ", timeout of " + timeout + " ms, interval of " + interval + " ms.";
                LOG.info(msg);
                status.setStatus(msg);
            }
            long sleepTime = 50L;
            Thread.sleep(50L);
            now = System.currentTimeMillis();
            slept = now - startTime;
            oldCount = count;
            count = this.countOfRegionServers();
            if (count == oldCount) continue;
            lastCountChange = now;
        }
        LOG.info("Finished waiting for region servers count to settle; checked in " + count + ", slept for " + slept + " ms," + " expecting minimum of " + minToStart + ", maximum of " + maxToStart + "," + " master is " + (this.master.isStopped() ? "stopped." : "running."));
    }

    public List<ServerName> getOnlineServersList() {
        return new ArrayList<ServerName>(this.onlineServers.keySet());
    }

    public List<ServerName> getDrainingServersList() {
        return new ArrayList<ServerName>(this.drainingServers);
    }

    Set<ServerName> getDeadNotExpiredServers() {
        return new HashSet<ServerName>(this.queuedDeadServers);
    }

    Map<ServerName, Boolean> getRequeuedDeadServers() {
        return Collections.unmodifiableMap(this.requeuedDeadServers);
    }

    public boolean isServerOnline(ServerName serverName) {
        return serverName != null && this.onlineServers.containsKey(serverName);
    }

    public synchronized boolean isServerDead(ServerName serverName) {
        return serverName == null || this.deadservers.isDeadServer(serverName) || this.queuedDeadServers.contains(serverName) || this.requeuedDeadServers.containsKey(serverName);
    }

    public void shutdownCluster() {
        this.clusterShutdown = true;
        this.master.stop("Cluster shutdown requested");
    }

    public boolean isClusterShutdown() {
        return this.clusterShutdown;
    }

    public void stop() {
        if (this.connection != null) {
            try {
                this.connection.close();
            }
            catch (IOException e) {
                LOG.error("Attempt to close connection to master failed", e);
            }
        }
    }

    public List<ServerName> createDestinationServersList(ServerName serverToExclude) {
        List<ServerName> drainingServersCopy;
        List<ServerName> destServers = this.getOnlineServersList();
        if (serverToExclude != null) {
            destServers.remove(serverToExclude);
        }
        if (!(drainingServersCopy = this.getDrainingServersList()).isEmpty()) {
            for (ServerName server : drainingServersCopy) {
                destServers.remove(server);
            }
        }
        this.removeDeadNotExpiredServers(destServers);
        return destServers;
    }

    public List<ServerName> createDestinationServersList() {
        return this.createDestinationServersList(null);
    }

    void removeDeadNotExpiredServers(List<ServerName> servers) {
        Set<ServerName> deadNotExpiredServersCopy = this.getDeadNotExpiredServers();
        if (!deadNotExpiredServersCopy.isEmpty()) {
            for (ServerName server : deadNotExpiredServersCopy) {
                LOG.debug("Removing dead but not expired server: " + server + " from eligible server pool.");
                servers.remove(server);
            }
        }
    }

    void clearDeadServersWithSameHostNameAndPortOfOnlineServer() {
        for (ServerName serverName : this.getOnlineServersList()) {
            this.deadservers.cleanAllPreviousInstances(serverName);
        }
    }
}

