/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master;

import com.google.common.collect.Sets;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InterruptedIOException;
import java.util.NavigableSet;
import java.util.TreeSet;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.hbase.CellUtil;
import org.apache.hadoop.hbase.HConstants;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.NamespaceDescriptor;
import org.apache.hadoop.hbase.NamespaceExistException;
import org.apache.hadoop.hbase.NamespaceNotFoundException;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.ZKNamespaceManager;
import org.apache.hadoop.hbase.catalog.MetaReader;
import org.apache.hadoop.hbase.client.Delete;
import org.apache.hadoop.hbase.client.Get;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.Put;
import org.apache.hadoop.hbase.client.Result;
import org.apache.hadoop.hbase.client.ResultScanner;
import org.apache.hadoop.hbase.constraint.ConstraintException;
import org.apache.hadoop.hbase.master.MasterServices;
import org.apache.hadoop.hbase.master.handler.CreateTableHandler;
import org.apache.hadoop.hbase.protobuf.ProtobufUtil;
import org.apache.hadoop.hbase.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.FSUtils;

@InterfaceAudience.Private
public class TableNamespaceManager {
    private static final Log LOG = LogFactory.getLog(TableNamespaceManager.class);
    private Configuration conf;
    private MasterServices masterServices;
    private HTable nsTable;
    private ZKNamespaceManager zkNamespaceManager;
    private boolean initialized;
    static final String NS_INIT_TIMEOUT = "hbase.master.namespace.init.timeout";
    static final int DEFAULT_NS_INIT_TIMEOUT = 60000;

    public TableNamespaceManager(MasterServices masterServices) {
        this.masterServices = masterServices;
        this.conf = masterServices.getConfiguration();
    }

    public void start() throws IOException {
        if (!MetaReader.tableExists(this.masterServices.getCatalogTracker(), TableName.NAMESPACE_TABLE_NAME)) {
            LOG.info("Namespace table not found. Creating...");
            this.createNamespaceTable(this.masterServices);
        }
        try {
            long startTime = EnvironmentEdgeManager.currentTimeMillis();
            int timeout = this.conf.getInt(NS_INIT_TIMEOUT, 60000);
            while (!this.isTableAssigned()) {
                if (EnvironmentEdgeManager.currentTimeMillis() - startTime + 100L > (long)timeout) {
                    LOG.warn("Timedout waiting for namespace table to be assigned.");
                    return;
                }
                Thread.sleep(100L);
            }
        }
        catch (InterruptedException e) {
            throw (InterruptedIOException)new InterruptedIOException().initCause(e);
        }
        this.isTableAvailableAndInitialized();
    }

    private synchronized HTable getNamespaceTable() throws IOException {
        if (!this.isTableAvailableAndInitialized()) {
            throw new IOException(this.getClass().getName() + " isn't ready to serve");
        }
        return this.nsTable;
    }

    public synchronized NamespaceDescriptor get(String name) throws IOException {
        if (!this.isTableAvailableAndInitialized()) {
            return null;
        }
        return this.zkNamespaceManager.get(name);
    }

    public synchronized void create(NamespaceDescriptor ns) throws IOException {
        this.create(this.getNamespaceTable(), ns);
    }

    public synchronized void update(NamespaceDescriptor ns) throws IOException {
        HTable table = this.getNamespaceTable();
        if (this.get(table, ns.getName()) == null) {
            throw new NamespaceNotFoundException(ns.getName());
        }
        this.upsert(table, ns);
    }

    private NamespaceDescriptor get(HTable table, String name) throws IOException {
        Result res = table.get(new Get(Bytes.toBytes(name)));
        if (res.isEmpty()) {
            return null;
        }
        byte[] val = CellUtil.cloneValue(res.getColumnLatestCell(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES, HTableDescriptor.NAMESPACE_COL_DESC_BYTES));
        return ProtobufUtil.toNamespaceDescriptor(HBaseProtos.NamespaceDescriptor.parseFrom(val));
    }

    private void create(HTable table, NamespaceDescriptor ns) throws IOException {
        if (this.get(table, ns.getName()) != null) {
            throw new NamespaceExistException(ns.getName());
        }
        FileSystem fs = this.masterServices.getMasterFileSystem().getFileSystem();
        fs.mkdirs(FSUtils.getNamespaceDir(this.masterServices.getMasterFileSystem().getRootDir(), ns.getName()));
        this.upsert(table, ns);
    }

    private void upsert(HTable table, NamespaceDescriptor ns) throws IOException {
        Put p = new Put(Bytes.toBytes(ns.getName()));
        p.addImmutable(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES, HTableDescriptor.NAMESPACE_COL_DESC_BYTES, ProtobufUtil.toProtoNamespaceDescriptor(ns).toByteArray());
        table.put(p);
        try {
            this.zkNamespaceManager.update(ns);
        }
        catch (IOException ex) {
            String msg = "Failed to update namespace information in ZK. Aborting.";
            LOG.fatal(msg, ex);
            this.masterServices.abort(msg, ex);
        }
    }

    public synchronized void remove(String name) throws IOException {
        int tableCount;
        if (this.get(name) == null) {
            throw new NamespaceNotFoundException(name);
        }
        if (NamespaceDescriptor.RESERVED_NAMESPACES.contains(name)) {
            throw new ConstraintException("Reserved namespace " + name + " cannot be removed.");
        }
        try {
            tableCount = this.masterServices.listTableDescriptorsByNamespace(name).size();
        }
        catch (FileNotFoundException fnfe) {
            throw new NamespaceNotFoundException(name);
        }
        if (tableCount > 0) {
            throw new ConstraintException("Only empty namespaces can be removed. Namespace " + name + " has " + tableCount + " tables");
        }
        Delete d = new Delete(Bytes.toBytes(name));
        this.getNamespaceTable().delete(d);
        this.zkNamespaceManager.remove(name);
        FileSystem fs = this.masterServices.getMasterFileSystem().getFileSystem();
        for (FileStatus status : fs.listStatus(FSUtils.getNamespaceDir(this.masterServices.getMasterFileSystem().getRootDir(), name))) {
            if (HConstants.HBASE_NON_TABLE_DIRS.contains(status.getPath().getName())) continue;
            throw new IOException("Namespace directory contains table dir: " + status.getPath());
        }
        if (!fs.delete(FSUtils.getNamespaceDir(this.masterServices.getMasterFileSystem().getRootDir(), name), true)) {
            throw new IOException("Failed to remove namespace: " + name);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized NavigableSet<NamespaceDescriptor> list() throws IOException {
        TreeSet<NamespaceDescriptor> ret = Sets.newTreeSet(NamespaceDescriptor.NAMESPACE_DESCRIPTOR_COMPARATOR);
        ResultScanner scanner = this.getNamespaceTable().getScanner(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES);
        try {
            for (Result r : scanner) {
                byte[] val = CellUtil.cloneValue(r.getColumnLatestCell(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES, HTableDescriptor.NAMESPACE_COL_DESC_BYTES));
                ret.add(ProtobufUtil.toNamespaceDescriptor(HBaseProtos.NamespaceDescriptor.parseFrom(val)));
            }
        }
        finally {
            scanner.close();
        }
        return ret;
    }

    private void createNamespaceTable(MasterServices masterServices) throws IOException {
        HRegionInfo[] newRegions = new HRegionInfo[]{new HRegionInfo(HTableDescriptor.NAMESPACE_TABLEDESC.getTableName(), null, null)};
        masterServices.getExecutorService().submit(new CreateTableHandler(masterServices, masterServices.getMasterFileSystem(), HTableDescriptor.NAMESPACE_TABLEDESC, masterServices.getConfiguration(), newRegions, masterServices).prepare());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized boolean isTableAvailableAndInitialized() throws IOException {
        if (this.initialized) {
            if (this.nsTable.getConnection().isClosed()) {
                this.nsTable = new HTable(this.conf, TableName.NAMESPACE_TABLE_NAME);
            }
            return true;
        }
        if (this.isTableAssigned()) {
            try {
                this.nsTable = new HTable(this.conf, TableName.NAMESPACE_TABLE_NAME);
                this.zkNamespaceManager = new ZKNamespaceManager(this.masterServices.getZooKeeper());
                this.zkNamespaceManager.start();
                if (this.get(this.nsTable, NamespaceDescriptor.DEFAULT_NAMESPACE.getName()) == null) {
                    this.create(this.nsTable, NamespaceDescriptor.DEFAULT_NAMESPACE);
                }
                if (this.get(this.nsTable, NamespaceDescriptor.SYSTEM_NAMESPACE.getName()) == null) {
                    this.create(this.nsTable, NamespaceDescriptor.SYSTEM_NAMESPACE);
                }
                ResultScanner scanner = this.nsTable.getScanner(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES);
                try {
                    for (Result result : scanner) {
                        byte[] val = CellUtil.cloneValue(result.getColumnLatest(HTableDescriptor.NAMESPACE_FAMILY_INFO_BYTES, HTableDescriptor.NAMESPACE_COL_DESC_BYTES));
                        NamespaceDescriptor ns = ProtobufUtil.toNamespaceDescriptor(HBaseProtos.NamespaceDescriptor.parseFrom(val));
                        this.zkNamespaceManager.update(ns);
                    }
                }
                finally {
                    scanner.close();
                }
                this.initialized = true;
                return true;
            }
            catch (IOException ie) {
                LOG.warn("Caught exception in initializing namespace table manager", ie);
                if (this.nsTable != null) {
                    this.nsTable.close();
                }
                throw ie;
            }
        }
        return false;
    }

    private boolean isTableAssigned() {
        return !this.masterServices.getAssignmentManager().getRegionStates().getRegionsOfTable(TableName.NAMESPACE_TABLE_NAME).isEmpty();
    }
}

