/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.lang.management.ManagementFactory;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.Executors;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.regionserver.MemStoreFlusher;
import org.apache.hadoop.hbase.regionserver.MemStoreLAB;
import org.apache.hadoop.util.StringUtils;

@InterfaceAudience.Private
public class MemStoreChunkPool {
    private static final Log LOG = LogFactory.getLog(MemStoreChunkPool.class);
    static final String CHUNK_POOL_MAXSIZE_KEY = "hbase.hregion.memstore.chunkpool.maxsize";
    static final String CHUNK_POOL_INITIALSIZE_KEY = "hbase.hregion.memstore.chunkpool.initialsize";
    static final float POOL_MAX_SIZE_DEFAULT = 0.0f;
    static final float POOL_INITIAL_SIZE_DEFAULT = 0.0f;
    private static MemStoreChunkPool globalInstance;
    static boolean chunkPoolDisabled;
    private final int maxCount;
    private final BlockingQueue<MemStoreLAB.Chunk> reclaimedChunks;
    private final int chunkSize;
    private final ScheduledExecutorService scheduleThreadPool;
    private static final int statThreadPeriod = 300;
    private AtomicLong createdChunkCount = new AtomicLong();
    private AtomicLong reusedChunkCount = new AtomicLong();

    MemStoreChunkPool(Configuration conf, int chunkSize, int maxCount, int initialCount) {
        this.maxCount = maxCount;
        this.chunkSize = chunkSize;
        this.reclaimedChunks = new LinkedBlockingQueue<MemStoreLAB.Chunk>();
        for (int i = 0; i < initialCount; ++i) {
            MemStoreLAB.Chunk chunk = new MemStoreLAB.Chunk(chunkSize);
            chunk.init();
            this.reclaimedChunks.add(chunk);
        }
        String n = Thread.currentThread().getName();
        this.scheduleThreadPool = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat(n + "-MemStoreChunkPool Statistics").setDaemon(true).build());
        this.scheduleThreadPool.scheduleAtFixedRate(new StatisticsThread(this), 300L, 300L, TimeUnit.SECONDS);
    }

    MemStoreLAB.Chunk getChunk() {
        MemStoreLAB.Chunk chunk = (MemStoreLAB.Chunk)this.reclaimedChunks.poll();
        if (chunk == null) {
            chunk = new MemStoreLAB.Chunk(this.chunkSize);
            this.createdChunkCount.incrementAndGet();
        } else {
            chunk.reset();
            this.reusedChunkCount.incrementAndGet();
        }
        return chunk;
    }

    void putbackChunks(BlockingQueue<MemStoreLAB.Chunk> chunks) {
        int maxNumToPutback = this.maxCount - this.reclaimedChunks.size();
        if (maxNumToPutback <= 0) {
            return;
        }
        chunks.drainTo(this.reclaimedChunks, maxNumToPutback);
    }

    void putbackChunk(MemStoreLAB.Chunk chunk) {
        if (this.reclaimedChunks.size() >= this.maxCount) {
            return;
        }
        this.reclaimedChunks.add(chunk);
    }

    int getPoolSize() {
        return this.reclaimedChunks.size();
    }

    void clearChunks() {
        this.reclaimedChunks.clear();
    }

    private void logStats() {
        if (!LOG.isDebugEnabled()) {
            return;
        }
        long created = this.createdChunkCount.get();
        long reused = this.reusedChunkCount.get();
        long total = created + reused;
        LOG.debug("Stats: current pool size=" + this.reclaimedChunks.size() + ",created chunk count=" + created + ",reused chunk count=" + reused + ",reuseRatio=" + (total == 0L ? "0" : StringUtils.formatPercent((float)reused / (float)total, 2)));
    }

    static synchronized MemStoreChunkPool getPool(Configuration conf) {
        if (globalInstance != null) {
            return globalInstance;
        }
        if (chunkPoolDisabled) {
            return null;
        }
        float poolSizePercentage = conf.getFloat(CHUNK_POOL_MAXSIZE_KEY, 0.0f);
        if (poolSizePercentage <= 0.0f) {
            chunkPoolDisabled = true;
            return null;
        }
        if ((double)poolSizePercentage > 1.0) {
            throw new IllegalArgumentException("hbase.hregion.memstore.chunkpool.maxsize must be between 0.0 and 1.0");
        }
        long heapMax = ManagementFactory.getMemoryMXBean().getHeapMemoryUsage().getMax();
        long globalMemStoreLimit = MemStoreFlusher.globalMemStoreLimit(heapMax, 0.4f, "hbase.regionserver.global.memstore.upperLimit", conf);
        int chunkSize = conf.getInt("hbase.hregion.memstore.mslab.chunksize", 0x200000);
        int maxCount = (int)((float)globalMemStoreLimit * poolSizePercentage / (float)chunkSize);
        float initialCountPercentage = conf.getFloat(CHUNK_POOL_INITIALSIZE_KEY, 0.0f);
        if ((double)initialCountPercentage > 1.0 || initialCountPercentage < 0.0f) {
            throw new IllegalArgumentException("hbase.hregion.memstore.chunkpool.initialsize must be between 0.0 and 1.0");
        }
        int initialCount = (int)(initialCountPercentage * (float)maxCount);
        LOG.info("Allocating MemStoreChunkPool with chunk size " + StringUtils.byteDesc(chunkSize) + ", max count " + maxCount + ", initial count " + initialCount);
        globalInstance = new MemStoreChunkPool(conf, chunkSize, maxCount, initialCount);
        return globalInstance;
    }

    static {
        chunkPoolDisabled = false;
    }

    private static class StatisticsThread
    extends Thread {
        MemStoreChunkPool mcp;

        public StatisticsThread(MemStoreChunkPool mcp) {
            super("MemStoreChunkPool.StatisticsThread");
            this.setDaemon(true);
            this.mcp = mcp;
        }

        @Override
        public void run() {
            this.mcp.logStats();
        }
    }
}

