/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.regionserver;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.regionserver.StoreFile;
import org.apache.hadoop.hbase.regionserver.StoreScanner;
import org.apache.hadoop.hbase.regionserver.StripeStoreFileManager;
import org.apache.hadoop.hbase.regionserver.compactions.Compactor;
import org.apache.hadoop.hbase.util.Bytes;

public abstract class StripeMultiFileWriter
implements Compactor.CellSink {
    private static final Log LOG = LogFactory.getLog(StripeMultiFileWriter.class);
    protected WriterFactory writerFactory;
    protected KeyValue.KVComparator comparator;
    protected List<StoreFile.Writer> existingWriters;
    protected List<byte[]> boundaries;
    protected StoreScanner sourceScanner;
    private boolean doWriteStripeMetadata = true;

    public void init(StoreScanner sourceScanner, WriterFactory factory, KeyValue.KVComparator comparator) throws IOException {
        this.writerFactory = factory;
        this.sourceScanner = sourceScanner;
        this.comparator = comparator;
    }

    public void setNoStripeMetadata() {
        this.doWriteStripeMetadata = false;
    }

    public List<Path> commitWriters(long maxSeqId, boolean isMajor) throws IOException {
        assert (this.existingWriters != null);
        this.commitWritersInternal();
        assert (this.boundaries.size() == this.existingWriters.size() + 1);
        LOG.debug((this.doWriteStripeMetadata ? "W" : "Not w") + "riting out metadata for " + this.existingWriters.size() + " writers");
        ArrayList<Path> paths = new ArrayList<Path>();
        for (int i = 0; i < this.existingWriters.size(); ++i) {
            StoreFile.Writer writer = this.existingWriters.get(i);
            if (writer == null) continue;
            if (this.doWriteStripeMetadata) {
                writer.appendFileInfo(StripeStoreFileManager.STRIPE_START_KEY, this.boundaries.get(i));
                writer.appendFileInfo(StripeStoreFileManager.STRIPE_END_KEY, this.boundaries.get(i + 1));
            }
            writer.appendMetadata(maxSeqId, isMajor);
            paths.add(writer.getPath());
            writer.close();
        }
        this.existingWriters = null;
        return paths;
    }

    public List<Path> abortWriters() {
        assert (this.existingWriters != null);
        ArrayList<Path> paths = new ArrayList<Path>();
        for (StoreFile.Writer writer : this.existingWriters) {
            try {
                paths.add(writer.getPath());
                writer.close();
            }
            catch (Exception ex) {
                LOG.error("Failed to close the writer after an unfinished compaction.", ex);
            }
        }
        this.existingWriters = null;
        return paths;
    }

    protected void sanityCheckLeft(byte[] left, byte[] row, int rowOffset, int rowLength) throws IOException {
        if (StripeStoreFileManager.OPEN_KEY != left && this.comparator.compareRows(row, rowOffset, rowLength, left, 0, left.length) < 0) {
            String error = "The first row is lower than the left boundary of [" + Bytes.toString(left) + "]: [" + Bytes.toString(row, rowOffset, rowLength) + "]";
            LOG.error(error);
            throw new IOException(error);
        }
    }

    protected void sanityCheckRight(byte[] right, byte[] row, int rowOffset, int rowLength) throws IOException {
        if (StripeStoreFileManager.OPEN_KEY != right && this.comparator.compareRows(row, rowOffset, rowLength, right, 0, right.length) >= 0) {
            String error = "The last row is higher or equal than the right boundary of [" + Bytes.toString(right) + "]: [" + Bytes.toString(row, rowOffset, rowLength) + "]";
            LOG.error(error);
            throw new IOException(error);
        }
    }

    protected abstract void commitWritersInternal() throws IOException;

    public static class SizeMultiWriter
    extends StripeMultiFileWriter {
        private int targetCount;
        private long targetKvs;
        private byte[] left;
        private byte[] right;
        private KeyValue lastKv;
        private StoreFile.Writer currentWriter;
        protected byte[] lastRowInCurrentWriter = null;
        private long kvsInCurrentWriter = 0L;
        private long kvsSeen = 0L;
        private long kvsSeenInPrevious = 0L;

        public SizeMultiWriter(int targetCount, long targetKvs, byte[] left, byte[] right) {
            this.targetCount = targetCount;
            this.targetKvs = targetKvs;
            this.left = left;
            this.right = right;
            int preallocate = Math.min(this.targetCount, 64);
            this.existingWriters = new ArrayList(preallocate);
            this.boundaries = new ArrayList(preallocate + 1);
        }

        @Override
        public void append(KeyValue kv) throws IOException {
            boolean doCreateWriter = false;
            if (this.currentWriter == null) {
                this.sanityCheckLeft(this.left, kv.getRowArray(), kv.getRowOffset(), kv.getRowLength());
                doCreateWriter = true;
            } else if (this.lastRowInCurrentWriter != null && !this.comparator.matchingRows(kv.getRowArray(), kv.getRowOffset(), kv.getRowLength(), this.lastRowInCurrentWriter, 0, this.lastRowInCurrentWriter.length)) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Stopping to use a writer after [" + Bytes.toString(this.lastRowInCurrentWriter) + "] row; wrote out " + this.kvsInCurrentWriter + " kvs");
                }
                this.lastRowInCurrentWriter = null;
                this.kvsInCurrentWriter = 0L;
                this.kvsSeenInPrevious += this.kvsSeen;
                doCreateWriter = true;
            }
            if (doCreateWriter) {
                byte[] boundary;
                byte[] byArray = boundary = this.existingWriters.isEmpty() ? this.left : kv.getRow();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Creating new writer starting at [" + Bytes.toString(boundary) + "]");
                }
                this.currentWriter = this.writerFactory.createWriter();
                this.boundaries.add(boundary);
                this.existingWriters.add(this.currentWriter);
            }
            this.currentWriter.append(kv);
            this.lastKv = kv;
            ++this.kvsInCurrentWriter;
            this.kvsSeen = this.kvsInCurrentWriter;
            if (this.sourceScanner != null) {
                this.kvsSeen = Math.max(this.kvsSeen, this.sourceScanner.getEstimatedNumberOfKvsScanned() - this.kvsSeenInPrevious);
            }
            if (this.lastRowInCurrentWriter == null && this.existingWriters.size() < this.targetCount && this.kvsSeen >= this.targetKvs) {
                this.lastRowInCurrentWriter = kv.getRow();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Preparing to start a new writer after [" + Bytes.toString(this.lastRowInCurrentWriter) + "] row; observed " + this.kvsSeen + " kvs and wrote out " + this.kvsInCurrentWriter + " kvs");
                }
            }
        }

        @Override
        protected void commitWritersInternal() throws IOException {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Stopping with " + this.kvsInCurrentWriter + " kvs in last writer" + (this.sourceScanner == null ? "" : "; observed estimated " + this.sourceScanner.getEstimatedNumberOfKvsScanned() + " KVs total"));
            }
            if (this.lastKv != null) {
                this.sanityCheckRight(this.right, this.lastKv.getRowArray(), this.lastKv.getRowOffset(), this.lastKv.getRowLength());
            }
            this.boundaries.add(this.right);
        }
    }

    public static class BoundaryMultiWriter
    extends StripeMultiFileWriter {
        private StoreFile.Writer currentWriter;
        private byte[] currentWriterEndKey;
        private KeyValue lastKv;
        private long kvsInCurrentWriter = 0L;
        private int majorRangeFromIndex = -1;
        private int majorRangeToIndex = -1;
        private boolean hasAnyWriter = false;

        public BoundaryMultiWriter(List<byte[]> targetBoundaries, byte[] majorRangeFrom, byte[] majorRangeTo) throws IOException {
            this.boundaries = targetBoundaries;
            this.existingWriters = new ArrayList(this.boundaries.size() - 1);
            assert (majorRangeFrom == null == (majorRangeTo == null));
            if (majorRangeFrom != null) {
                this.majorRangeFromIndex = majorRangeFrom == StripeStoreFileManager.OPEN_KEY ? 0 : Collections.binarySearch(this.boundaries, majorRangeFrom, Bytes.BYTES_COMPARATOR);
                int n = this.majorRangeToIndex = majorRangeTo == StripeStoreFileManager.OPEN_KEY ? this.boundaries.size() : Collections.binarySearch(this.boundaries, majorRangeTo, Bytes.BYTES_COMPARATOR);
                if (this.majorRangeFromIndex < 0 || this.majorRangeToIndex < 0) {
                    throw new IOException("Major range does not match writer boundaries: [" + Bytes.toString(majorRangeFrom) + "] [" + Bytes.toString(majorRangeTo) + "]; from " + this.majorRangeFromIndex + " to " + this.majorRangeToIndex);
                }
            }
        }

        @Override
        public void append(KeyValue kv) throws IOException {
            if (this.currentWriter == null && this.existingWriters.isEmpty()) {
                this.sanityCheckLeft((byte[])this.boundaries.get(0), kv.getRowArray(), kv.getRowOffset(), kv.getRowLength());
            }
            this.prepareWriterFor(kv);
            this.currentWriter.append(kv);
            this.lastKv = kv;
            ++this.kvsInCurrentWriter;
        }

        private boolean isKvAfterCurrentWriter(KeyValue kv) {
            return this.currentWriterEndKey != StripeStoreFileManager.OPEN_KEY && this.comparator.compareRows(kv.getRowArray(), kv.getRowOffset(), kv.getRowLength(), this.currentWriterEndKey, 0, this.currentWriterEndKey.length) >= 0;
        }

        @Override
        protected void commitWritersInternal() throws IOException {
            this.stopUsingCurrentWriter();
            while (this.existingWriters.size() < this.boundaries.size() - 1) {
                this.createEmptyWriter();
            }
            if (this.lastKv != null) {
                this.sanityCheckRight((byte[])this.boundaries.get(this.boundaries.size() - 1), this.lastKv.getRowArray(), this.lastKv.getRowOffset(), this.lastKv.getRowLength());
            }
        }

        private void prepareWriterFor(KeyValue kv) throws IOException {
            if (this.currentWriter != null && !this.isKvAfterCurrentWriter(kv)) {
                return;
            }
            this.stopUsingCurrentWriter();
            while (this.isKvAfterCurrentWriter(kv)) {
                this.checkCanCreateWriter();
                this.createEmptyWriter();
            }
            this.checkCanCreateWriter();
            this.hasAnyWriter = true;
            this.currentWriter = this.writerFactory.createWriter();
            this.existingWriters.add(this.currentWriter);
        }

        private void createEmptyWriter() throws IOException {
            int index = this.existingWriters.size();
            boolean isInMajorRange = index >= this.majorRangeFromIndex && index < this.majorRangeToIndex;
            boolean isLastWriter = !this.hasAnyWriter && index == this.boundaries.size() - 2;
            boolean needEmptyFile = isInMajorRange || isLastWriter;
            this.existingWriters.add(needEmptyFile ? this.writerFactory.createWriter() : null);
            this.hasAnyWriter |= needEmptyFile;
            this.currentWriterEndKey = this.existingWriters.size() + 1 == this.boundaries.size() ? null : (byte[])this.boundaries.get(this.existingWriters.size() + 1);
        }

        private void checkCanCreateWriter() throws IOException {
            int maxWriterCount = this.boundaries.size() - 1;
            assert (this.existingWriters.size() <= maxWriterCount);
            if (this.existingWriters.size() >= maxWriterCount) {
                throw new IOException("Cannot create any more writers (created " + this.existingWriters.size() + " out of " + maxWriterCount + " - row might be out of range of all valid writers");
            }
        }

        private void stopUsingCurrentWriter() {
            if (this.currentWriter != null) {
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Stopping to use a writer after [" + Bytes.toString(this.currentWriterEndKey) + "] row; wrote out " + this.kvsInCurrentWriter + " kvs");
                }
                this.kvsInCurrentWriter = 0L;
            }
            this.currentWriter = null;
            this.currentWriterEndKey = this.existingWriters.size() + 1 == this.boundaries.size() ? null : (byte[])this.boundaries.get(this.existingWriters.size() + 1);
        }
    }

    public static interface WriterFactory {
        public StoreFile.Writer createWriter() throws IOException;
    }
}

