/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.snapshot;

import java.io.IOException;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.logging.impl.Log4JLogger;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.HBaseTestingUtility;
import org.apache.hadoop.hbase.HRegionInfo;
import org.apache.hadoop.hbase.HTableDescriptor;
import org.apache.hadoop.hbase.LargeTests;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.TableNotFoundException;
import org.apache.hadoop.hbase.client.HBaseAdmin;
import org.apache.hadoop.hbase.client.HTable;
import org.apache.hadoop.hbase.client.ScannerCallable;
import org.apache.hadoop.hbase.ipc.RpcClient;
import org.apache.hadoop.hbase.ipc.RpcServer;
import org.apache.hadoop.hbase.master.HMaster;
import org.apache.hadoop.hbase.protobuf.generated.HBaseProtos;
import org.apache.hadoop.hbase.regionserver.ConstantSizeRegionSplitPolicy;
import org.apache.hadoop.hbase.regionserver.HRegionFileSystem;
import org.apache.hadoop.hbase.snapshot.ClientSnapshotDescriptionUtils;
import org.apache.hadoop.hbase.snapshot.SnapshotCreationException;
import org.apache.hadoop.hbase.snapshot.SnapshotDescriptionUtils;
import org.apache.hadoop.hbase.snapshot.SnapshotTestingUtils;
import org.apache.hadoop.hbase.util.Bytes;
import org.apache.hadoop.hbase.util.FSTableDescriptors;
import org.apache.hadoop.hbase.util.FSUtils;
import org.apache.hadoop.hbase.util.JVMClusterUtil;
import org.apache.log4j.Level;
import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;
import org.junit.experimental.categories.Category;

@Category(value={LargeTests.class})
public class TestFlushSnapshotFromClient {
    private static final Log LOG = LogFactory.getLog(TestFlushSnapshotFromClient.class);
    private static final HBaseTestingUtility UTIL = new HBaseTestingUtility();
    private static final int NUM_RS = 2;
    private static final String STRING_TABLE_NAME = "test";
    private static final byte[] TEST_FAM = Bytes.toBytes("fam");
    private static final byte[] TEST_QUAL = Bytes.toBytes("q");
    private static final TableName TABLE_NAME = TableName.valueOf("test");
    private final int DEFAULT_NUM_ROWS = 100;

    @BeforeClass
    public static void setupCluster() throws Exception {
        ((Log4JLogger)RpcServer.LOG).getLogger().setLevel(Level.ALL);
        ((Log4JLogger)RpcClient.LOG).getLogger().setLevel(Level.ALL);
        ((Log4JLogger)ScannerCallable.LOG).getLogger().setLevel(Level.ALL);
        TestFlushSnapshotFromClient.setupConf(UTIL.getConfiguration());
        UTIL.startMiniCluster(2);
    }

    private static void setupConf(Configuration conf) {
        conf.setInt("hbase.regionsever.info.port", -1);
        conf.setInt("hbase.hregion.memstore.flush.size", 25000);
        conf.setInt("hbase.hstore.compaction.min", 10);
        conf.setInt("hbase.hstore.compactionThreshold", 10);
        conf.setInt("hbase.hstore.blockingStoreFiles", 12);
        conf.setBoolean("hbase.snapshot.enabled", true);
        conf.set("hbase.regionserver.region.split.policy", ConstantSizeRegionSplitPolicy.class.getName());
    }

    @Before
    public void setup() throws Exception {
        SnapshotTestingUtils.createTable(UTIL, TABLE_NAME, new byte[][]{TEST_FAM});
    }

    @After
    public void tearDown() throws Exception {
        UTIL.deleteTable(TABLE_NAME);
        SnapshotTestingUtils.deleteAllSnapshots(UTIL.getHBaseAdmin());
        SnapshotTestingUtils.deleteArchiveDirectory(UTIL);
    }

    @AfterClass
    public static void cleanupTest() throws Exception {
        try {
            UTIL.shutdownMiniCluster();
        }
        catch (Exception e) {
            LOG.warn("failure shutting down cluster", e);
        }
    }

    @Test(timeout=300000L)
    public void testFlushTableSnapshot() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        HTable table = new HTable(UTIL.getConfiguration(), TABLE_NAME);
        SnapshotTestingUtils.loadData(UTIL, table, 100, (byte[][])new byte[][]{TEST_FAM});
        HashSet<String> snapshotServers = new HashSet<String>();
        List<JVMClusterUtil.RegionServerThread> servers = UTIL.getMiniHBaseCluster().getLiveRegionServerThreads();
        for (JVMClusterUtil.RegionServerThread server : servers) {
            if (server.getRegionServer().getOnlineRegions(TABLE_NAME).size() <= 0) continue;
            snapshotServers.add(server.getRegionServer().getServerName().toString());
        }
        LOG.debug("FS state before snapshot:");
        FSUtils.logFileSystemState(UTIL.getTestFileSystem(), FSUtils.getRootDir(UTIL.getConfiguration()), LOG);
        String snapshotString = "offlineTableSnapshot";
        byte[] snapshot = Bytes.toBytes(snapshotString);
        admin.snapshot(snapshotString, STRING_TABLE_NAME, HBaseProtos.SnapshotDescription.Type.FLUSH);
        LOG.debug("Snapshot completed.");
        List<HBaseProtos.SnapshotDescription> snapshots = SnapshotTestingUtils.assertOneSnapshotThatMatches(admin, snapshot, TABLE_NAME);
        FileSystem fs = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getFileSystem();
        Path rootDir = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getRootDir();
        LOG.debug("FS state after snapshot:");
        FSUtils.logFileSystemState(UTIL.getTestFileSystem(), FSUtils.getRootDir(UTIL.getConfiguration()), LOG);
        SnapshotTestingUtils.confirmSnapshotValid(snapshots.get(0), TABLE_NAME, TEST_FAM, rootDir, admin, fs, false, new Path(rootDir, "WALs"), snapshotServers);
    }

    @Test(timeout=300000L)
    public void testFlushTableSnapshotWithProcedure() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        HTable table = new HTable(UTIL.getConfiguration(), TABLE_NAME);
        SnapshotTestingUtils.loadData(UTIL, table, 100, (byte[][])new byte[][]{TEST_FAM});
        HashSet<String> snapshotServers = new HashSet<String>();
        List<JVMClusterUtil.RegionServerThread> servers = UTIL.getMiniHBaseCluster().getLiveRegionServerThreads();
        for (JVMClusterUtil.RegionServerThread server : servers) {
            if (server.getRegionServer().getOnlineRegions(TABLE_NAME).size() <= 0) continue;
            snapshotServers.add(server.getRegionServer().getServerName().toString());
        }
        LOG.debug("FS state before snapshot:");
        FSUtils.logFileSystemState(UTIL.getTestFileSystem(), FSUtils.getRootDir(UTIL.getConfiguration()), LOG);
        String snapshotString = "offlineTableSnapshot";
        byte[] snapshot = Bytes.toBytes(snapshotString);
        HashMap<String, String> props = new HashMap<String, String>();
        props.put("table", STRING_TABLE_NAME);
        admin.execProcedure("online-snapshot", snapshotString, props);
        LOG.debug("Snapshot completed.");
        List<HBaseProtos.SnapshotDescription> snapshots = SnapshotTestingUtils.assertOneSnapshotThatMatches(admin, snapshot, TABLE_NAME);
        FileSystem fs = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getFileSystem();
        Path rootDir = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getRootDir();
        LOG.debug("FS state after snapshot:");
        FSUtils.logFileSystemState(UTIL.getTestFileSystem(), FSUtils.getRootDir(UTIL.getConfiguration()), LOG);
        SnapshotTestingUtils.confirmSnapshotValid(snapshots.get(0), TABLE_NAME, TEST_FAM, rootDir, admin, fs, false, new Path(rootDir, "WALs"), snapshotServers);
    }

    @Test(timeout=300000L)
    public void testSnapshotFailsOnNonExistantTable() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        String tableName = "_not_a_table";
        boolean fail = false;
        do {
            try {
                admin.getTableDescriptor(Bytes.toBytes(tableName));
                fail = true;
                LOG.error("Table:" + tableName + " already exists, checking a new name");
                tableName = tableName + "!";
            }
            catch (TableNotFoundException e) {
                fail = false;
            }
        } while (fail);
        try {
            admin.snapshot("fail", tableName, HBaseProtos.SnapshotDescription.Type.FLUSH);
            Assert.fail((String)"Snapshot succeeded even though there is not table.");
        }
        catch (SnapshotCreationException e) {
            LOG.info("Correctly failed to snapshot a non-existant table:" + e.getMessage());
        }
    }

    @Test(timeout=300000L)
    public void testAsyncFlushSnapshot() throws Exception {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        HBaseProtos.SnapshotDescription snapshot = HBaseProtos.SnapshotDescription.newBuilder().setName("asyncSnapshot").setTable(TABLE_NAME.getNameAsString()).setType(HBaseProtos.SnapshotDescription.Type.FLUSH).build();
        admin.takeSnapshotAsync(snapshot);
        HMaster master = UTIL.getMiniHBaseCluster().getMaster();
        SnapshotTestingUtils.waitForSnapshotToComplete(master, snapshot, 200L);
        LOG.info(" === Async Snapshot Completed ===");
        FSUtils.logFileSystemState(UTIL.getTestFileSystem(), FSUtils.getRootDir(UTIL.getConfiguration()), LOG);
        SnapshotTestingUtils.assertOneSnapshotThatMatches(admin, snapshot);
    }

    @Test(timeout=300000L)
    public void testSnapshotStateAfterMerge() throws Exception {
        int numRows = 100;
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        SnapshotTestingUtils.loadData(UTIL, TABLE_NAME, numRows, (byte[][])new byte[][]{TEST_FAM});
        String snapshotBeforeMergeName = "snapshotBeforeMerge";
        admin.snapshot(snapshotBeforeMergeName, STRING_TABLE_NAME, HBaseProtos.SnapshotDescription.Type.FLUSH);
        String cloneBeforeMergeName = "cloneBeforeMerge";
        admin.cloneSnapshot(snapshotBeforeMergeName, cloneBeforeMergeName);
        SnapshotTestingUtils.waitForTableToBeOnline(UTIL, TableName.valueOf(cloneBeforeMergeName));
        List<HRegionInfo> regions = admin.getTableRegions(TABLE_NAME);
        Collections.sort(regions, new Comparator<HRegionInfo>(){

            @Override
            public int compare(HRegionInfo r1, HRegionInfo r2) {
                return Bytes.compareTo(r1.getStartKey(), r2.getStartKey());
            }
        });
        int numRegions = admin.getTableRegions(TABLE_NAME).size();
        int numRegionsAfterMerge = numRegions - 2;
        admin.mergeRegions(regions.get(1).getEncodedNameAsBytes(), regions.get(2).getEncodedNameAsBytes(), true);
        admin.mergeRegions(regions.get(5).getEncodedNameAsBytes(), regions.get(6).getEncodedNameAsBytes(), true);
        this.waitRegionsAfterMerge(numRegionsAfterMerge);
        Assert.assertEquals((long)numRegionsAfterMerge, (long)admin.getTableRegions(TABLE_NAME).size());
        String cloneAfterMergeName = "cloneAfterMerge";
        admin.cloneSnapshot(snapshotBeforeMergeName, cloneAfterMergeName);
        SnapshotTestingUtils.waitForTableToBeOnline(UTIL, TableName.valueOf(cloneAfterMergeName));
        SnapshotTestingUtils.verifyRowCount(UTIL, TABLE_NAME, numRows);
        SnapshotTestingUtils.verifyRowCount(UTIL, TableName.valueOf(cloneBeforeMergeName), numRows);
        SnapshotTestingUtils.verifyRowCount(UTIL, TableName.valueOf(cloneAfterMergeName), numRows);
        UTIL.deleteTable(cloneAfterMergeName);
        UTIL.deleteTable(cloneBeforeMergeName);
    }

    @Test(timeout=300000L)
    public void testTakeSnapshotAfterMerge() throws Exception {
        int numRows = 100;
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        SnapshotTestingUtils.loadData(UTIL, TABLE_NAME, numRows, (byte[][])new byte[][]{TEST_FAM});
        List<HRegionInfo> regions = admin.getTableRegions(TABLE_NAME);
        Collections.sort(regions, new Comparator<HRegionInfo>(){

            @Override
            public int compare(HRegionInfo r1, HRegionInfo r2) {
                return Bytes.compareTo(r1.getStartKey(), r2.getStartKey());
            }
        });
        int numRegions = admin.getTableRegions(TABLE_NAME).size();
        int numRegionsAfterMerge = numRegions - 2;
        admin.mergeRegions(regions.get(1).getEncodedNameAsBytes(), regions.get(2).getEncodedNameAsBytes(), true);
        admin.mergeRegions(regions.get(5).getEncodedNameAsBytes(), regions.get(6).getEncodedNameAsBytes(), true);
        this.waitRegionsAfterMerge(numRegionsAfterMerge);
        Assert.assertEquals((long)numRegionsAfterMerge, (long)admin.getTableRegions(TABLE_NAME).size());
        String snapshotName = "snapshotAfterMerge";
        SnapshotTestingUtils.snapshot(admin, snapshotName, STRING_TABLE_NAME, HBaseProtos.SnapshotDescription.Type.FLUSH, 3);
        String cloneName = "cloneMerge";
        admin.cloneSnapshot(snapshotName, cloneName);
        SnapshotTestingUtils.waitForTableToBeOnline(UTIL, TableName.valueOf(cloneName));
        SnapshotTestingUtils.verifyRowCount(UTIL, TABLE_NAME, numRows);
        SnapshotTestingUtils.verifyRowCount(UTIL, TableName.valueOf(cloneName), numRows);
        UTIL.deleteTable(cloneName);
    }

    @Test(timeout=300000L)
    public void testFlushCreateListDestroy() throws Exception {
        LOG.debug("------- Starting Snapshot test -------------");
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        SnapshotTestingUtils.loadData(UTIL, TABLE_NAME, 100, (byte[][])new byte[][]{TEST_FAM});
        String snapshotName = "flushSnapshotCreateListDestroy";
        admin.snapshot(snapshotName, STRING_TABLE_NAME, HBaseProtos.SnapshotDescription.Type.FLUSH);
        this.logFSTree(FSUtils.getRootDir(UTIL.getConfiguration()));
        List<HBaseProtos.SnapshotDescription> snapshots = SnapshotTestingUtils.assertOneSnapshotThatMatches(admin, snapshotName, TABLE_NAME);
        FileSystem fs = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getFileSystem();
        Path rootDir = UTIL.getHBaseCluster().getMaster().getMasterFileSystem().getRootDir();
        Path snapshotDir = SnapshotDescriptionUtils.getCompletedSnapshotDir(snapshots.get(0), rootDir);
        Assert.assertTrue((boolean)fs.exists(snapshotDir));
        FSUtils.logFileSystemState(UTIL.getTestFileSystem(), snapshotDir, LOG);
        Path snapshotinfo = new Path(snapshotDir, ".snapshotinfo");
        Assert.assertTrue((boolean)fs.exists(snapshotinfo));
        HTableDescriptor desc = FSTableDescriptors.getTableDescriptorFromFs(fs, rootDir, TABLE_NAME);
        HTableDescriptor snapshotDesc = FSTableDescriptors.getTableDescriptorFromFs(fs, new Path(SnapshotDescriptionUtils.getSnapshotsDir(rootDir), snapshotName));
        Assert.assertEquals((Object)desc, (Object)snapshotDesc);
        List<HRegionInfo> regions = admin.getTableRegions(TABLE_NAME);
        Assert.assertTrue((regions.size() > 1 ? 1 : 0) != 0);
        for (HRegionInfo info : regions) {
            String regionName = info.getEncodedName();
            Path regionDir = new Path(snapshotDir, regionName);
            HRegionInfo snapshotRegionInfo = HRegionFileSystem.loadRegionInfoFileContent(fs, regionDir);
            Assert.assertEquals((Object)info, (Object)snapshotRegionInfo);
            Path familyDir = new Path(regionDir, Bytes.toString(TEST_FAM));
            Assert.assertTrue((String)("Missing region " + Bytes.toString(snapshotRegionInfo.getStartKey())), (boolean)fs.exists(familyDir));
            Assert.assertTrue((fs.listStatus(familyDir).length > 0 ? 1 : 0) != 0);
        }
    }

    @Test(timeout=300000L)
    public void testConcurrentSnapshottingAttempts() throws IOException, InterruptedException {
        int i;
        String STRING_TABLE2_NAME = "test2";
        TableName TABLE2_NAME = TableName.valueOf("test2");
        int ssNum = 20;
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        SnapshotTestingUtils.assertNoSnapshots(admin);
        SnapshotTestingUtils.createTable(UTIL, TABLE2_NAME, new byte[][]{TEST_FAM});
        SnapshotTestingUtils.loadData(UTIL, TABLE_NAME, 100, (byte[][])new byte[][]{TEST_FAM});
        SnapshotTestingUtils.loadData(UTIL, TABLE2_NAME, 100, (byte[][])new byte[][]{TEST_FAM});
        final CountDownLatch toBeSubmitted = new CountDownLatch(ssNum);
        HBaseProtos.SnapshotDescription[] descs = new HBaseProtos.SnapshotDescription[ssNum];
        for (i = 0; i < ssNum; ++i) {
            HBaseProtos.SnapshotDescription.Builder builder = HBaseProtos.SnapshotDescription.newBuilder();
            builder.setTable((i % 2 == 0 ? TABLE_NAME : TABLE2_NAME).getNameAsString());
            builder.setName("ss" + i);
            builder.setType(HBaseProtos.SnapshotDescription.Type.FLUSH);
            descs[i] = builder.build();
        }
        for (i = 0; i < ssNum; ++i) {
            class SSRunnable
            implements Runnable {
                HBaseProtos.SnapshotDescription ss;

                SSRunnable(HBaseProtos.SnapshotDescription ss) {
                    this.ss = ss;
                }

                @Override
                public void run() {
                    try {
                        HBaseAdmin admin = UTIL.getHBaseAdmin();
                        LOG.info("Submitting snapshot request: " + ClientSnapshotDescriptionUtils.toString(this.ss));
                        admin.takeSnapshotAsync(this.ss);
                    }
                    catch (Exception e) {
                        LOG.info("Exception during snapshot request: " + ClientSnapshotDescriptionUtils.toString(this.ss) + ".  This is ok, we expect some", e);
                    }
                    LOG.info("Submitted snapshot request: " + ClientSnapshotDescriptionUtils.toString(this.ss));
                    toBeSubmitted.countDown();
                }
            }
            new Thread(new SSRunnable(descs[i])).start();
        }
        toBeSubmitted.await();
        while (true) {
            int doneCount = 0;
            for (HBaseProtos.SnapshotDescription ss : descs) {
                try {
                    if (!admin.isSnapshotFinished(ss)) continue;
                    ++doneCount;
                }
                catch (Exception e) {
                    LOG.warn("Got an exception when checking for snapshot " + ss.getName(), e);
                    ++doneCount;
                }
            }
            if (doneCount == descs.length) break;
            Thread.sleep(100L);
        }
        this.logFSTree(FSUtils.getRootDir(UTIL.getConfiguration()));
        List<HBaseProtos.SnapshotDescription> taken = admin.listSnapshots();
        int takenSize = taken.size();
        LOG.info("Taken " + takenSize + " snapshots:  " + taken);
        Assert.assertTrue((String)"We expect at least 1 request to be rejected because of we concurrently issued many requests", (takenSize < ssNum && takenSize > 0 ? 1 : 0) != 0);
        int t1SnapshotsCount = 0;
        int t2SnapshotsCount = 0;
        for (HBaseProtos.SnapshotDescription ss : taken) {
            if (TableName.valueOf(ss.getTable()).equals(TABLE_NAME)) {
                ++t1SnapshotsCount;
                continue;
            }
            if (!TableName.valueOf(ss.getTable()).equals(TABLE2_NAME)) continue;
            ++t2SnapshotsCount;
        }
        Assert.assertTrue((String)"We expect at least 1 snapshot of table1 ", (t1SnapshotsCount > 0 ? 1 : 0) != 0);
        Assert.assertTrue((String)"We expect at least 1 snapshot of table2 ", (t2SnapshotsCount > 0 ? 1 : 0) != 0);
        UTIL.deleteTable(TABLE2_NAME);
    }

    private void logFSTree(Path root) throws IOException {
        FSUtils.logFileSystemState(UTIL.getDFSCluster().getFileSystem(), root, LOG);
    }

    private void waitRegionsAfterMerge(long numRegionsAfterMerge) throws IOException, InterruptedException {
        HBaseAdmin admin = UTIL.getHBaseAdmin();
        long startTime = System.currentTimeMillis();
        while ((long)admin.getTableRegions(TABLE_NAME).size() != numRegionsAfterMerge && System.currentTimeMillis() - startTime <= 15000L) {
            Thread.sleep(100L);
        }
        SnapshotTestingUtils.waitForTableToBeOnline(UTIL, TABLE_NAME);
    }
}

