/*
 * Decompiled with CFR 0.152.
 */
package org.apache.phoenix.mapreduce;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import java.io.IOException;
import java.io.StringReader;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.util.Iterator;
import java.util.List;
import javax.annotation.Nullable;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.KeyValue;
import org.apache.hadoop.hbase.io.ImmutableBytesWritable;
import org.apache.hadoop.hbase.util.Pair;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.Text;
import org.apache.hadoop.mapreduce.Mapper;
import org.apache.hadoop.util.ReflectionUtils;
import org.apache.phoenix.jdbc.PhoenixConnection;
import org.apache.phoenix.jdbc.PhoenixDriver;
import org.apache.phoenix.mapreduce.ImportPreUpsertKeyValueProcessor;
import org.apache.phoenix.util.ColumnInfo;
import org.apache.phoenix.util.PhoenixRuntime;
import org.apache.phoenix.util.csv.CsvUpsertExecutor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CsvToKeyValueMapper
extends Mapper<LongWritable, Text, ImmutableBytesWritable, KeyValue> {
    private static final Logger LOG = LoggerFactory.getLogger(CsvToKeyValueMapper.class);
    private static final String COUNTER_GROUP_NAME = "Phoenix MapReduce Import";
    public static final String UPSERT_HOOK_CLASS_CONFKEY = "phoenix.mapreduce.import.kvprocessor";
    public static final String FIELD_DELIMITER_CONFKEY = "phoenix.mapreduce.import.fielddelimiter";
    public static final String ARRAY_DELIMITER_CONFKEY = "phoenix.mapreduce.import.arraydelimiter";
    public static final String TABLE_NAME_CONFKEY = "phoenix.mapreduce.import.tablename";
    public static final String COLUMN_INFO_CONFKEY = "phoenix.mapreduce.import.columninfos";
    public static final String IGNORE_INVALID_ROW_CONFKEY = "phoenix.mapreduce.import.ignoreinvalidrow";
    private PhoenixConnection conn;
    private CsvUpsertExecutor csvUpsertExecutor;
    private MapperUpsertListener upsertListener;
    private CsvLineParser csvLineParser;
    private ImportPreUpsertKeyValueProcessor preUpdateProcessor;

    @Override
    protected void setup(Mapper.Context context) throws IOException, InterruptedException {
        Configuration conf = context.getConfiguration();
        String jdbcUrl = CsvToKeyValueMapper.getJdbcUrl(conf);
        LOG.info("Connection with driver {} with url {}", (Object)PhoenixDriver.class.getName(), (Object)jdbcUrl);
        try {
            this.conn = (PhoenixConnection)DriverManager.getConnection(jdbcUrl);
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
        this.upsertListener = new MapperUpsertListener(context, conf.getBoolean(IGNORE_INVALID_ROW_CONFKEY, true));
        this.csvUpsertExecutor = this.buildUpsertExecutor(conf);
        this.csvLineParser = new CsvLineParser(conf.get(FIELD_DELIMITER_CONFKEY).charAt(0));
        this.preUpdateProcessor = CsvToKeyValueMapper.loadPreUpsertProcessor(conf);
    }

    @Override
    protected void map(LongWritable key, Text value, Mapper.Context context) throws IOException, InterruptedException {
        ImmutableBytesWritable outputKey = new ImmutableBytesWritable();
        try {
            CSVRecord csvRecord = null;
            try {
                csvRecord = this.csvLineParser.parse(value.toString());
            }
            catch (IOException e) {
                context.getCounter(COUNTER_GROUP_NAME, "CSV Parser errors").increment(1L);
            }
            if (csvRecord == null) {
                context.getCounter(COUNTER_GROUP_NAME, "Empty records").increment(1L);
                return;
            }
            this.csvUpsertExecutor.execute(ImmutableList.of(csvRecord));
            Iterator<Pair<byte[], List<KeyValue>>> uncommittedDataIterator = PhoenixRuntime.getUncommittedDataIterator(this.conn);
            while (uncommittedDataIterator.hasNext()) {
                Pair<byte[], List<KeyValue>> kvPair = uncommittedDataIterator.next();
                List<KeyValue> keyValueList = kvPair.getSecond();
                keyValueList = this.preUpdateProcessor.preUpsert(kvPair.getFirst(), keyValueList);
                for (KeyValue kv : keyValueList) {
                    outputKey.set(kv.getBuffer(), kv.getRowOffset(), kv.getRowLength());
                    context.write(outputKey, kv);
                }
            }
            this.conn.rollback();
        }
        catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    protected void cleanup(Mapper.Context context) throws IOException, InterruptedException {
        try {
            this.conn.close();
        }
        catch (SQLException e) {
            throw new RuntimeException(e);
        }
    }

    @VisibleForTesting
    static ImportPreUpsertKeyValueProcessor loadPreUpsertProcessor(Configuration conf) {
        Class<ImportPreUpsertKeyValueProcessor> processorClass = null;
        try {
            processorClass = conf.getClass(UPSERT_HOOK_CLASS_CONFKEY, DefaultImportPreUpsertKeyValueProcessor.class, ImportPreUpsertKeyValueProcessor.class);
        }
        catch (Exception e) {
            throw new IllegalStateException("Couldn't load upsert hook class", e);
        }
        return ReflectionUtils.newInstance(processorClass, conf);
    }

    @VisibleForTesting
    static String getJdbcUrl(Configuration conf) {
        String zkQuorum = conf.get("hbase.zookeeper.quorum");
        if (zkQuorum == null) {
            throw new IllegalStateException("hbase.zookeeper.quorum is not configured");
        }
        return "jdbc:phoenix:" + zkQuorum;
    }

    @VisibleForTesting
    CsvUpsertExecutor buildUpsertExecutor(Configuration conf) {
        String tableName = conf.get(TABLE_NAME_CONFKEY);
        String arraySeparator = conf.get(ARRAY_DELIMITER_CONFKEY, ":");
        Preconditions.checkNotNull(tableName, "table name is not configured");
        List<ColumnInfo> columnInfoList = CsvToKeyValueMapper.buildColumnInfoList(conf);
        return CsvUpsertExecutor.create(this.conn, tableName, columnInfoList, this.upsertListener, arraySeparator);
    }

    @VisibleForTesting
    static void configureColumnInfoList(Configuration conf, List<ColumnInfo> columnInfoList) {
        conf.set(COLUMN_INFO_CONFKEY, Joiner.on("|").useForNull("").join(columnInfoList));
    }

    @VisibleForTesting
    static List<ColumnInfo> buildColumnInfoList(Configuration conf) {
        return Lists.newArrayList(Iterables.transform(Splitter.on("|").split(conf.get(COLUMN_INFO_CONFKEY)), new Function<String, ColumnInfo>(){

            @Override
            @Nullable
            public ColumnInfo apply(@Nullable String input) {
                if (input.isEmpty()) {
                    return null;
                }
                return ColumnInfo.fromString(input);
            }
        }));
    }

    public static class DefaultImportPreUpsertKeyValueProcessor
    implements ImportPreUpsertKeyValueProcessor {
        @Override
        public List<KeyValue> preUpsert(byte[] rowKey, List<KeyValue> keyValues) {
            return keyValues;
        }
    }

    @VisibleForTesting
    static class CsvLineParser {
        private final CSVFormat csvFormat;

        CsvLineParser(char fieldDelimiter) {
            this.csvFormat = CSVFormat.newFormat(fieldDelimiter);
        }

        public CSVRecord parse(String input) throws IOException {
            CSVParser csvParser = new CSVParser(new StringReader(input), this.csvFormat);
            return Iterables.getFirst(csvParser, null);
        }
    }

    @VisibleForTesting
    static class MapperUpsertListener
    implements CsvUpsertExecutor.UpsertListener {
        private final Mapper.Context context;
        private final boolean ignoreRecordErrors;

        private MapperUpsertListener(Mapper.Context context, boolean ignoreRecordErrors) {
            this.context = context;
            this.ignoreRecordErrors = ignoreRecordErrors;
        }

        @Override
        public void upsertDone(long upsertCount) {
            this.context.getCounter(CsvToKeyValueMapper.COUNTER_GROUP_NAME, "Upserts Done").increment(1L);
        }

        @Override
        public void errorOnRecord(CSVRecord csvRecord, String errorMessage) {
            LOG.error("Error on record {}: {}", csvRecord, (Object)errorMessage);
            this.context.getCounter(CsvToKeyValueMapper.COUNTER_GROUP_NAME, "Errors on records").increment(1L);
            if (!this.ignoreRecordErrors) {
                throw new RuntimeException("Error on record, " + errorMessage + ", " + "record =" + csvRecord);
            }
        }
    }
}

