/*
 * Decompiled with CFR 0.152.
 */
package org.apache.phoenix.util;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.io.File;
import java.io.Reader;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import org.apache.commons.csv.CSVFormat;
import org.apache.commons.csv.CSVParser;
import org.apache.commons.csv.CSVRecord;
import org.apache.phoenix.exception.SQLExceptionCode;
import org.apache.phoenix.exception.SQLExceptionInfo;
import org.apache.phoenix.jdbc.PhoenixConnection;
import org.apache.phoenix.schema.PDataType;
import org.apache.phoenix.util.ColumnInfo;
import org.apache.phoenix.util.StringUtil;
import org.apache.phoenix.util.csv.CsvUpsertExecutor;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CSVCommonsLoader {
    private static final Logger LOG = LoggerFactory.getLogger(CSVCommonsLoader.class);
    public static final String DEFAULT_ARRAY_ELEMENT_SEPARATOR = ":";
    private final PhoenixConnection conn;
    private final String tableName;
    private final List<String> columns;
    private final boolean isStrict;
    boolean userSuppliedMetaCharacters = false;
    private final List<String> customMetaCharacters;
    private PhoenixHeaderSource headerSource = PhoenixHeaderSource.FROM_TABLE;
    private final CSVFormat format;
    private final Map<String, Character> ctrlTable = new HashMap<String, Character>(){
        {
            this.put("1", Character.valueOf('\u0001'));
            this.put("2", Character.valueOf('\u0002'));
            this.put("3", Character.valueOf('\u0003'));
            this.put("4", Character.valueOf('\u0004'));
            this.put("5", Character.valueOf('\u0005'));
            this.put("6", Character.valueOf('\u0006'));
            this.put("7", Character.valueOf('\u0007'));
            this.put("8", Character.valueOf('\b'));
            this.put("9", Character.valueOf('\t'));
        }
    };
    private final String arrayElementSeparator;

    public CSVCommonsLoader(PhoenixConnection conn, String tableName, List<String> columns, boolean isStrict) {
        this(conn, tableName, columns, isStrict, null, DEFAULT_ARRAY_ELEMENT_SEPARATOR);
    }

    public CSVCommonsLoader(PhoenixConnection conn, String tableName, List<String> columns, boolean isStrict, List<String> customMetaCharacters, String arrayElementSeparator) {
        this.conn = conn;
        this.tableName = tableName;
        this.columns = columns;
        this.isStrict = isStrict;
        this.customMetaCharacters = customMetaCharacters;
        if (customMetaCharacters == null || customMetaCharacters.size() == 0) {
            this.userSuppliedMetaCharacters = false;
        } else if (customMetaCharacters.size() == 3) {
            this.userSuppliedMetaCharacters = true;
        } else {
            throw new IllegalArgumentException(String.format("customMetaCharacters must have no elements or three elements. Supplied value is %s", CSVCommonsLoader.buildStringFromList(customMetaCharacters)));
        }
        if (columns != null && !columns.isEmpty()) {
            this.headerSource = PhoenixHeaderSource.SUPPLIED_BY_USER;
        } else if (columns != null && columns.isEmpty()) {
            this.headerSource = PhoenixHeaderSource.IN_LINE;
        }
        this.arrayElementSeparator = arrayElementSeparator;
        this.format = this.buildFormat();
    }

    public CSVFormat getFormat() {
        return this.format;
    }

    private CSVFormat buildFormat() {
        CSVFormat format = CSVFormat.DEFAULT.withIgnoreEmptyLines(true);
        if (this.userSuppliedMetaCharacters) {
            String delimiter = this.customMetaCharacters.get(0);
            String quote = this.customMetaCharacters.get(1);
            String escape = this.customMetaCharacters.get(2);
            if (!"0".equals(delimiter)) {
                format = format.withDelimiter(this.getCustomMetaCharacter(delimiter));
            }
            if (!"0".equals(quote)) {
                format = format.withQuoteChar(this.getCustomMetaCharacter(quote));
            }
            if (!"0".equals(quote)) {
                format = format.withEscape(this.getCustomMetaCharacter(escape));
            }
        }
        switch (this.headerSource) {
            case FROM_TABLE: {
                break;
            }
            case IN_LINE: {
                format = format.withHeader(new String[0]);
                break;
            }
            case SUPPLIED_BY_USER: {
                format = format.withHeader(this.columns.toArray(new String[this.columns.size()]));
                break;
            }
            default: {
                throw new RuntimeException("Header source was unable to be inferred.");
            }
        }
        return format;
    }

    public char getCustomMetaCharacter(String field) {
        if (this.ctrlTable.containsKey(field)) {
            return this.ctrlTable.get(field).charValue();
        }
        return field.charAt(0);
    }

    public void upsert(String fileName) throws Exception {
        CSVParser parser = CSVParser.parse(new File(fileName), this.format);
        this.upsert(parser);
    }

    public void upsert(Reader reader) throws Exception {
        CSVParser parser = new CSVParser(reader, this.format);
        this.upsert(parser);
    }

    private static <T> String buildStringFromList(List<T> list) {
        return Joiner.on(", ").useForNull("null").join(list);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void upsert(CSVParser csvParser) throws Exception {
        List<ColumnInfo> columnInfoList = this.buildColumnInfoList(csvParser);
        boolean wasAutoCommit = this.conn.getAutoCommit();
        try {
            this.conn.setAutoCommit(false);
            long start = System.currentTimeMillis();
            CsvUpsertListener upsertListener = new CsvUpsertListener(this.conn, this.conn.getMutateBatchSize());
            CsvUpsertExecutor csvUpsertExecutor = CsvUpsertExecutor.create(this.conn, this.tableName, columnInfoList, upsertListener, this.arrayElementSeparator);
            csvUpsertExecutor.execute(csvParser);
            csvUpsertExecutor.close();
            this.conn.commit();
            double elapsedDuration = (double)(System.currentTimeMillis() - start) / 1000.0;
            System.out.println("CSV Upsert complete. " + upsertListener.getTotalUpsertCount() + " rows upserted");
            System.out.println("Time: " + elapsedDuration + " sec(s)\n");
        }
        finally {
            if (csvParser != null) {
                csvParser.close();
            }
            if (wasAutoCommit) {
                this.conn.setAutoCommit(true);
            }
        }
    }

    private List<ColumnInfo> buildColumnInfoList(CSVParser parser) throws SQLException {
        List<String> columns = this.columns;
        switch (this.headerSource) {
            case FROM_TABLE: {
                System.out.println(String.format("csv columns from database.", new Object[0]));
                break;
            }
            case IN_LINE: {
                columns = new ArrayList<String>();
                for (String colName : parser.getHeaderMap().keySet()) {
                    columns.add(colName);
                }
                System.out.println(String.format("csv columns from header line. length=%s, %s", columns.size(), CSVCommonsLoader.buildStringFromList(columns)));
                break;
            }
            case SUPPLIED_BY_USER: {
                System.out.println(String.format("csv columns from user. length=%s, %s", columns.size(), CSVCommonsLoader.buildStringFromList(columns)));
                break;
            }
            default: {
                throw new IllegalStateException("parser has unknown column source.");
            }
        }
        return CSVCommonsLoader.generateColumnInfo(this.conn, this.tableName, columns, this.isStrict);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static List<ColumnInfo> generateColumnInfo(Connection conn, String tableName, List<String> columns, boolean strict) throws SQLException {
        LinkedHashMap<String, Integer> columnNameToTypeMap = Maps.newLinkedHashMap();
        DatabaseMetaData dbmd = conn.getMetaData();
        int unfoundColumnCount = 0;
        String escapedTableName = StringUtil.escapeLike(tableName);
        String[] schemaAndTable = escapedTableName.split("\\.");
        ResultSet rs = null;
        try {
            rs = dbmd.getColumns(null, schemaAndTable.length == 1 ? "" : schemaAndTable[0], schemaAndTable.length == 1 ? escapedTableName : schemaAndTable[1], null);
            while (rs.next()) {
                String sqlTypeName = rs.getString(6);
                columnNameToTypeMap.put(rs.getString(4), PDataType.fromSqlTypeName(sqlTypeName).getSqlType());
            }
        }
        finally {
            if (rs != null) {
                rs.close();
            }
        }
        ArrayList<ColumnInfo> columnInfoList = Lists.newArrayList();
        if (columns == null) {
            for (Map.Entry entry : columnNameToTypeMap.entrySet()) {
                columnInfoList.add(new ColumnInfo((String)entry.getKey(), (Integer)entry.getValue()));
            }
        } else {
            for (int i = 0; i < columns.size(); ++i) {
                String columnName = columns.get(i).trim();
                Integer sqlType = (Integer)columnNameToTypeMap.get(columnName);
                if (sqlType == null) {
                    if (strict) {
                        throw new SQLExceptionInfo.Builder(SQLExceptionCode.COLUMN_NOT_FOUND).setColumnName(columnName).setTableName(tableName).build().buildException();
                    }
                    ++unfoundColumnCount;
                    continue;
                }
                columnInfoList.add(new ColumnInfo(columnName, sqlType));
            }
            if (unfoundColumnCount == columns.size()) {
                throw new SQLExceptionInfo.Builder(SQLExceptionCode.COLUMN_NOT_FOUND).setColumnName(Arrays.toString(columns.toArray(new String[0]))).setTableName(tableName).build().buildException();
            }
        }
        return columnInfoList;
    }

    static class CsvUpsertListener
    implements CsvUpsertExecutor.UpsertListener {
        private final PhoenixConnection conn;
        private final int upsertBatchSize;
        private long totalUpserts = 0L;

        CsvUpsertListener(PhoenixConnection conn, int upsertBatchSize) {
            this.conn = conn;
            this.upsertBatchSize = upsertBatchSize;
        }

        @Override
        public void upsertDone(long upsertCount) {
            this.totalUpserts = upsertCount;
            if (upsertCount % (long)this.upsertBatchSize == 0L) {
                if (upsertCount % 1000L == 0L) {
                    LOG.info("Processed upsert #{}", upsertCount);
                }
                try {
                    LOG.info("Committing after {} records", upsertCount);
                    this.conn.commit();
                }
                catch (SQLException e) {
                    throw new RuntimeException(e);
                }
            }
        }

        @Override
        public void errorOnRecord(CSVRecord csvRecord, String errorMessage) {
            LOG.error("Error upserting record {}: {}", csvRecord, (Object)errorMessage);
        }

        public long getTotalUpsertCount() {
            return this.totalUpserts;
        }
    }

    public static enum PhoenixHeaderSource {
        FROM_TABLE,
        IN_LINE,
        SUPPLIED_BY_USER;

    }
}

