/*
 * Decompiled with CFR 0.152.
 */
package org.andresoviedo.app.model3D.services.collada.loader;

import android.opengl.Matrix;
import android.util.Log;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import org.andresoviedo.app.model3D.services.collada.entities.JointData;
import org.andresoviedo.app.model3D.services.collada.entities.MeshData;
import org.andresoviedo.app.model3D.services.collada.entities.SkeletonData;
import org.andresoviedo.app.model3D.services.collada.entities.SkinningData;
import org.andresoviedo.app.model3D.services.collada.entities.Vector2f;
import org.andresoviedo.app.model3D.services.collada.entities.Vector3f;
import org.andresoviedo.app.model3D.services.collada.entities.Vector4f;
import org.andresoviedo.app.model3D.services.collada.entities.Vertex;
import org.andresoviedo.app.model3D.services.collada.entities.VertexSkinData;
import org.andresoviedo.app.util.xml.XmlNode;

public class GeometryLoader {
    private static final float[] CORRECTION = new float[16];
    private final XmlNode geometryNode;
    private final XmlNode materialsData;
    private final XmlNode effectsData;
    private final XmlNode imagesNode;
    private Map<String, SkinningData> skinningDataMap;
    private SkeletonData skeletonData;
    private float[] verticesArray;
    private float[] normalsArray;
    private float[] texturesArray;
    private int[] indicesArray;
    private int[] jointIdsArray;
    private float[] weightsArray;
    private FloatBuffer colorsBuffer;
    List<Vertex> vertices = new ArrayList<Vertex>();
    List<Vector2f> textures = new ArrayList<Vector2f>();
    List<Vector3f> normals = new ArrayList<Vector3f>();
    List<Integer> indices = new ArrayList<Integer>();
    List<float[]> colors = new ArrayList<float[]>();

    public GeometryLoader(XmlNode geometryNode, XmlNode materialsNode, XmlNode effectsNode, XmlNode imagesNode, Map<String, SkinningData> skinningData, SkeletonData skeletonData) {
        this.skinningDataMap = skinningData;
        this.geometryNode = geometryNode;
        this.materialsData = materialsNode;
        this.imagesNode = imagesNode;
        this.effectsData = effectsNode;
        this.skeletonData = skeletonData;
    }

    public List<MeshData> extractModelData() {
        ArrayList<MeshData> ret = new ArrayList<MeshData>();
        for (XmlNode geometry : this.geometryNode.getChildren("geometry")) {
            this.vertices.clear();
            this.normals.clear();
            this.textures.clear();
            this.indices.clear();
            this.colors.clear();
            String geometryId = geometry.getAttribute("id");
            Log.d((String)"GeometryLoader", (String)("Loading geometry '" + geometryId + "'"));
            XmlNode meshData = geometry.getChild("mesh");
            this.readRawData(meshData, geometryId);
            List<XmlNode> polys = meshData.getChildren("polylist");
            String texture = null;
            for (XmlNode poly : polys) {
                String material = poly.getAttribute("material");
                float[] color = this.getMaterialColor(material);
                texture = color == null ? this.getTexture(material) : null;
                this.assembleVertices(poly, color);
            }
            List<XmlNode> triangless = meshData.getChildren("triangles");
            for (XmlNode triangles : triangless) {
                String material = triangles.getAttribute("material");
                float[] color = this.getMaterialColor(material);
                texture = color == null ? this.getTexture(material) : null;
                this.assembleVertices(triangles, color);
            }
            this.removeUnusedVertices();
            this.initArrays(geometryId);
            this.convertDataToArrays();
            this.convertIndicesListToArray();
            ret.add(new MeshData(geometryId, this.verticesArray, this.texturesArray, this.normalsArray, this.colorsBuffer, texture, this.indicesArray, this.jointIdsArray, this.weightsArray));
        }
        return ret;
    }

    private void readRawData(XmlNode meshData, String geometryId) {
        this.readPositions(meshData, geometryId);
        XmlNode polylist = meshData.getChild("polylist");
        XmlNode triangles = meshData.getChild("triangles");
        String normalsId = null;
        String texCoordsId = null;
        if (polylist != null) {
            normalsId = polylist.getChildWithAttribute("input", "semantic", "NORMAL").getAttribute("source").substring(1);
            XmlNode childWithAttribute = polylist.getChildWithAttribute("input", "semantic", "TEXCOORD");
            if (childWithAttribute != null) {
                texCoordsId = childWithAttribute.getAttribute("source").substring(1);
            }
        } else if (triangles != null) {
            normalsId = triangles.getChildWithAttribute("input", "semantic", "NORMAL").getAttribute("source").substring(1);
            XmlNode childWithAttribute = triangles.getChildWithAttribute("input", "semantic", "TEXCOORD");
            if (childWithAttribute != null) {
                texCoordsId = childWithAttribute.getAttribute("source").substring(1);
            }
        }
        this.readNormals(meshData, normalsId);
        if (texCoordsId != null) {
            this.readTextureCoords(meshData, texCoordsId);
        } else {
            Log.i((String)"GeometryLoader", (String)("No texture data found for '" + geometryId + "'"));
        }
    }

    private JointData getJointData(JointData jointData, String geometryId) {
        if (geometryId.equals(jointData.meshId)) {
            return jointData;
        }
        for (JointData childJointData : jointData.children) {
            JointData candidate = this.getJointData(childJointData, geometryId);
            if (candidate == null) continue;
            return candidate;
        }
        return null;
    }

    private void readPositions(XmlNode meshData, String geometryId) {
        String positionsId = meshData.getChild("vertices").getChild("input").getAttribute("source").substring(1);
        XmlNode positionsData = meshData.getChildWithAttribute("source", "id", positionsId).getChild("float_array");
        int count = Integer.parseInt(positionsData.getAttribute("count"));
        String[] posData = positionsData.getData().trim().split("\\s+");
        for (int i = 0; i < count / 3; ++i) {
            JointData jointData;
            VertexSkinData weightsData;
            float x = Float.parseFloat(posData[i * 3]);
            float y = Float.parseFloat(posData[i * 3 + 1]);
            float z = Float.parseFloat(posData[i * 3 + 2]);
            Vector4f position = new Vector4f(x, y, z, 1.0f);
            float[] positionV = new float[4];
            Matrix.multiplyMV((float[])positionV, (int)0, (float[])CORRECTION, (int)0, (float[])position.toArray(), (int)0);
            position = new Vector4f(positionV);
            VertexSkinData vertexSkinData = weightsData = this.skinningDataMap != null && this.skinningDataMap.containsKey(geometryId) ? this.skinningDataMap.get((Object)geometryId).verticesSkinData.get(this.vertices.size()) : null;
            if (weightsData == null && this.skeletonData != null && (jointData = this.getJointData(this.skeletonData.headJoint, geometryId)) != null) {
                weightsData = new VertexSkinData();
                weightsData.addJointEffect(jointData.index, 1.0f);
                weightsData.limitJointNumber(3);
            }
            this.vertices.add(new Vertex(this.vertices.size(), new Vector3f(position.x, position.y, position.z), weightsData));
        }
        Log.i((String)"GeometryLoader", (String)("Vertex count: " + this.vertices.size()));
    }

    private void readNormals(XmlNode meshData, String normalsId) {
        XmlNode normalsData = meshData.getChildWithAttribute("source", "id", normalsId).getChild("float_array");
        int count = Integer.parseInt(normalsData.getAttribute("count"));
        String[] normData = normalsData.getData().trim().split("\\s+");
        for (int i = 0; i < count / 3; ++i) {
            float x = Float.parseFloat(normData[i * 3]);
            float y = Float.parseFloat(normData[i * 3 + 1]);
            float z = Float.parseFloat(normData[i * 3 + 2]);
            Vector4f norm = new Vector4f(x, y, z, 0.0f);
            float[] normV = new float[4];
            Matrix.multiplyMV((float[])normV, (int)0, (float[])CORRECTION, (int)0, (float[])norm.toArray(), (int)0);
            norm = new Vector4f(normV);
            this.normals.add(new Vector3f(norm.x, norm.y, norm.z));
        }
    }

    private void readTextureCoords(XmlNode meshData, String texCoordsId) {
        XmlNode texCoordsData = meshData.getChildWithAttribute("source", "id", texCoordsId).getChild("float_array");
        int count = Integer.parseInt(texCoordsData.getAttribute("count"));
        String[] texData = texCoordsData.getData().trim().split("\\s+");
        for (int i = 0; i < count / 2; ++i) {
            float s = Float.parseFloat(texData[i * 2]);
            float t = Float.parseFloat(texData[i * 2 + 1]);
            this.textures.add(new Vector2f(s, t));
        }
    }

    private void assembleVertices(XmlNode poly, float[] color) {
        int typeCount = 0;
        for (XmlNode input : poly.getChildren("input")) {
            int offset = Integer.valueOf(input.getAttribute("offset")) + 1;
            if (offset <= typeCount) continue;
            typeCount = offset;
        }
        Log.i((String)"GeometryLoader", (String)("Loading polygon. Stride: " + typeCount));
        XmlNode texcoordSemantic = poly.getChildWithAttribute("input", "semantic", "TEXCOORD");
        int texcoordOffset = texcoordSemantic != null ? Integer.parseInt(texcoordSemantic.getAttribute("offset")) : -1;
        String[] indexData = poly.getChild("p").getData().trim().split("\\s+");
        for (int i = 0; i < indexData.length / typeCount; ++i) {
            int positionIndex = Integer.parseInt(indexData[i * typeCount]);
            int normalIndex = Integer.parseInt(indexData[i * typeCount + 1]);
            int texCoordIndex = -1;
            if (texcoordOffset != -1) {
                texCoordIndex = Integer.parseInt(indexData[i * typeCount + texcoordOffset]);
            }
            this.processVertex(positionIndex, normalIndex, texCoordIndex, color);
        }
    }

    private Vertex processVertex(int posIndex, int normIndex, int texIndex, float[] color) {
        Vertex currentVertex = this.vertices.get(posIndex);
        if (!currentVertex.isSet()) {
            currentVertex.setTextureIndex(texIndex);
            currentVertex.setNormalIndex(normIndex);
            this.indices.add(posIndex);
            if (color != null) {
                this.colors.add(color);
            }
            return currentVertex;
        }
        return this.dealWithAlreadyProcessedVertex(currentVertex, texIndex, normIndex, color);
    }

    private float[] getMaterialColor(String material) {
        try {
            XmlNode diffuse;
            XmlNode colorNode;
            XmlNode materialNode = this.materialsData.getChildWithAttribute("material", "id", material);
            XmlNode instanceEffectNode = materialNode.getChild("instance_effect");
            String instanceEffectId = instanceEffectNode.getAttribute("url").substring(1);
            XmlNode effect = this.effectsData.getChildWithAttribute("effect", "id", instanceEffectId);
            XmlNode profile_common = effect.getChild("profile_COMMON");
            XmlNode technique = profile_common.getChild("technique");
            XmlNode lambert = technique.getChild("lambert");
            if (lambert == null) {
                lambert = technique.getChild("phong");
            }
            if ((colorNode = (diffuse = lambert.getChild("diffuse")).getChild("color")) != null) {
                String[] color = colorNode.getData().trim().split("\\s+");
                return new float[]{Float.valueOf(color[0]).floatValue(), Float.valueOf(color[1]).floatValue(), Float.valueOf(color[2]).floatValue(), Float.valueOf(color[3]).floatValue()};
            }
            return null;
        }
        catch (Exception ex) {
            Log.e((String)"GeometryLoader", (String)("No color found for material '" + material + "'"), (Throwable)ex);
            return null;
        }
    }

    private String getTexture(String material) {
        try {
            XmlNode diffuse;
            XmlNode textureNode;
            XmlNode materialNode = this.materialsData.getChildWithAttribute("material", "id", material);
            XmlNode instanceEffectNode = materialNode.getChild("instance_effect");
            String instanceEffectId = instanceEffectNode.getAttribute("url").substring(1);
            XmlNode effect = this.effectsData.getChildWithAttribute("effect", "id", instanceEffectId);
            XmlNode profile_common = effect.getChild("profile_COMMON");
            XmlNode technique = profile_common.getChild("technique");
            XmlNode lambert = technique.getChild("lambert");
            if (lambert == null) {
                lambert = technique.getChild("phong");
            }
            if ((textureNode = (diffuse = lambert.getChild("diffuse")).getChild("texture")) != null) {
                String texture = textureNode.getAttribute("texture");
                XmlNode newParamNode = profile_common.getChildWithAttribute("newparam", "sid", texture);
                String surface = newParamNode.getChild("sampler2D").getChild("source").getData();
                newParamNode = profile_common.getChildWithAttribute("newparam", "sid", surface);
                String imageRef = newParamNode.getChildWithAttribute("surface", "type", "2D").getChild("init_from").getData();
                String image = this.imagesNode.getChildWithAttribute("image", "id", imageRef).getChild("init_from").getData();
                return image;
            }
            return null;
        }
        catch (Exception ex) {
            Log.e((String)"GeometryLoader", (String)("No texture found for material '" + material + "'"), (Throwable)ex);
            return null;
        }
    }

    private int[] convertIndicesListToArray() {
        this.indicesArray = new int[this.indices.size()];
        for (int i = 0; i < this.indicesArray.length; ++i) {
            this.indicesArray[i] = this.indices.get(i);
        }
        return this.indicesArray;
    }

    private float convertDataToArrays() {
        float furthestPoint = 0.0f;
        for (int i = 0; i < this.vertices.size(); ++i) {
            Vertex currentVertex = this.vertices.get(i);
            if (currentVertex.getLength() > furthestPoint) {
                furthestPoint = currentVertex.getLength();
            }
            Vector3f position = currentVertex.getPosition();
            if (this.textures != null && !this.textures.isEmpty()) {
                Vector2f textureCoord = this.textures.get(currentVertex.getTextureIndex());
                this.texturesArray[i * 2] = textureCoord.x;
                this.texturesArray[i * 2 + 1] = 1.0f - textureCoord.y;
            }
            Vector3f normalVector = this.normals.get(currentVertex.getNormalIndex());
            this.verticesArray[i * 3] = position.x;
            this.verticesArray[i * 3 + 1] = position.y;
            this.verticesArray[i * 3 + 2] = position.z;
            this.normalsArray[i * 3] = normalVector.x;
            this.normalsArray[i * 3 + 1] = normalVector.y;
            this.normalsArray[i * 3 + 2] = normalVector.z;
            VertexSkinData weights = currentVertex.getWeightsData();
            if (weights == null) continue;
            this.jointIdsArray[i * 3] = weights.jointIds.get(0);
            this.jointIdsArray[i * 3 + 1] = weights.jointIds.get(1);
            this.jointIdsArray[i * 3 + 2] = weights.jointIds.get(2);
            this.weightsArray[i * 3] = weights.weights.get(0).floatValue();
            this.weightsArray[i * 3 + 1] = weights.weights.get(1).floatValue();
            this.weightsArray[i * 3 + 2] = weights.weights.get(2).floatValue();
        }
        for (float[] color : this.colors) {
            this.colorsBuffer.put(color);
        }
        return furthestPoint;
    }

    private Vertex dealWithAlreadyProcessedVertex(Vertex previousVertex, int newTextureIndex, int newNormalIndex, float[] color) {
        if (previousVertex.hasSameTextureAndNormal(newTextureIndex, newNormalIndex)) {
            this.indices.add(previousVertex.getIndex());
            return previousVertex;
        }
        Vertex anotherVertex = previousVertex.getDuplicateVertex();
        if (anotherVertex != null) {
            return this.dealWithAlreadyProcessedVertex(anotherVertex, newTextureIndex, newNormalIndex, color);
        }
        Vertex duplicateVertex = new Vertex(this.vertices.size(), previousVertex.getPosition(), previousVertex.getWeightsData());
        duplicateVertex.setTextureIndex(newTextureIndex);
        duplicateVertex.setNormalIndex(newNormalIndex);
        previousVertex.setDuplicateVertex(duplicateVertex);
        this.vertices.add(duplicateVertex);
        this.indices.add(duplicateVertex.getIndex());
        if (color != null) {
            this.colors.add(color);
        }
        return duplicateVertex;
    }

    private void initArrays(String geometryId) {
        this.verticesArray = new float[this.vertices.size() * 3];
        if (this.textures != null && !this.textures.isEmpty()) {
            this.texturesArray = new float[this.vertices.size() * 2];
        }
        this.normalsArray = new float[this.vertices.size() * 3];
        if (this.skinningDataMap != null && this.skinningDataMap.containsKey(geometryId) || this.vertices.get(0).getWeightsData() != null) {
            this.jointIdsArray = new int[this.vertices.size() * 3];
            this.weightsArray = new float[this.vertices.size() * 3];
        }
        if (!this.colors.isEmpty()) {
            this.colorsBuffer = GeometryLoader.createNativeByteBuffer(this.colors.size() * 4 * 4).asFloatBuffer();
        }
    }

    private void removeUnusedVertices() {
        for (Vertex vertex : this.vertices) {
            vertex.averageTangents();
            if (vertex.isSet()) continue;
            vertex.setTextureIndex(0);
            vertex.setNormalIndex(0);
        }
    }

    private static ByteBuffer createNativeByteBuffer(int length) {
        ByteBuffer bb = ByteBuffer.allocateDirect(length);
        bb.order(ByteOrder.nativeOrder());
        return bb;
    }

    static {
        Matrix.setIdentityM((float[])CORRECTION, (int)0);
        Matrix.rotateM((float[])CORRECTION, (int)0, (float[])CORRECTION, (int)0, (float)-90.0f, (float)1.0f, (float)0.0f, (float)0.0f);
    }
}

